(define-module (tilemap)
  #:use-module (ice-9 match)
  #:use-module (ice-9 vlist)
  #:use-module (srfi srfi-1)              ;lists?
  #:use-module (srfi srfi-9)              ;records?
  #:use-module (srfi srfi-26)             ;partial?
  #:use-module (sdl2)
  #:use-module (sly game)
  #:use-module (sly window)
  #:use-module (sly signal)
  #:use-module (sly utils)
  #:use-module (sly render)
  #:use-module (sly render camera)
  #:use-module (sly render color)
  #:use-module (sly render mesh)
  #:use-module (sly render shader)
  #:use-module (sly render sprite)
  #:use-module (sly render texture)
  #:use-module (sly render tileset)
  #:use-module (sly render tile-map)
  #:use-module (sly math rect)
  #:use-module (sly math vector)
  #:use-module (tiled)
  #:use-module (collision)
  #:use-module (utils)
  #:use-module (fade)
  #:use-module (sound)
  #:use-module (character)
  #:use-module (player)
  #:export (tilemap-file
            tilemap
            tiles-per-x
            tiles-per-y
            tile-width
            tile-height
            map-current-name
            map-loops
            map-centers
            map-clamps
            map-width
            map-height
            map-dimensions
            map-use-spawn
            tile-dimensions

            map-background
            map-background-position
            map-characters

            ;; character shit
            action-on-next-tile
            locked-action
            sleep-timer-offset

            needs-to-spawn
            change-map
            collides-with-map
            handle-collision-with-portals
            colliding-logic-object))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DEFINITIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-signal tilemap-file "rooms/room.tmx")
(define-signal tilemap
  (signal-let ((game-started? game-started?)
               (tilemap-file tilemap-file))
              (and game-started?
                   tilemap-file
                   (load-tmx tilemap-file load-tileset))))
(define tile-width 16)
(define tile-height 16)
(define-signal map-loops
  (signal-map (lambda (tilemap)
                (and tilemap (tilemap-property tilemap 'loops)))
              tilemap))
(define-signal map-centers #f)
(define-signal map-clamps #f)
(define-signal map-use-spawn "desk-awake")
(define-signal needs-to-spawn #t)
(define-signal needs-to-spawn-npcs #t)
(define (simplified-map-name file)
  (string->symbol (basename (string-downcase file) ".tmx")))
(define-signal map-current-name
  (signal-map (lambda (file)
                (simplified-map-name file))
              tilemap-file))
(define-signal tiles-per-x
  (signal-map (lambda (tilemap)
                (or (and tilemap
                         (tilemap-tiles-per-x tilemap))
                    1))
              tilemap))
(define-signal tiles-per-y
  (signal-map (lambda (tilemap)
                (or (and tilemap
                         (tilemap-tiles-per-y tilemap))
                    1))
              tilemap))
(define-signal map-width
  (signal-map (lambda (per-x) (* tile-width per-x))
              tiles-per-x))
(define-signal map-height
  (signal-map (lambda (per-y) (* tile-height per-y))
              tiles-per-y))
(define-signal map-dimensions
  (signal-map vector2 map-width map-height))
(define tile-dimensions (vector2 tile-width tile-height))

(define-signal map-background
  (signal-let ((needs-to-spawn needs-to-spawn)
               (tilemap tilemap)
               (game-started? game-started?))
              (if (and game-started? tilemap)
                  (let ()
                    (define property (tilemap-get-map-property tilemap 'background))
                    (define path (and property (property-value property)))
                    (warn "path" path)
                    (and path (load-sprite path)))
                  #f)))
(define-signal map-background-delta
  (signal-let ((needs-to-spawn needs-to-spawn)
               (tilemap tilemap)
               (game-started? game-started?))
              (if (and game-started? tilemap)
                  (let ()
                    (define property-x (tilemap-get-map-property tilemap 'background-delta-x))
                    (define property-y (tilemap-get-map-property tilemap 'background-delta-y))
                    (define x (or (property-value property-x) 2))
                    (define y (or (property-value property-y) 0))
                    (vector2 x y))
                  (vector2 0 0))))

(define-signal map-background-position
  (signal-fold (lambda (time delta old)
                 ;; TODO make it time dependant
                 ;; TODO loop around
                 (define added (v+ old delta))
                 (define w (* (signal-ref map-width) *scale*))
                 (define h (* (signal-ref map-height) *scale*))
                 (define loops (signal-ref map-loops))
                 (cond ((and loops (> (vx added) w)) (v- added (vector2 w 0)))
                       ((and loops (< (vx added) 0)) (v+ added (vector2 w 0)))
                       ((and loops (> (vy added) h)) (v- added (vector2 0 h)))
                       ((and loops (< (vy added) 0)) (v+ added (vector2 0 h)))
                       ;; TODO use background size for add/sub
                       ((> (vx added) (+ w w)) (v- added (vector2 (* w 3) 0)))
                       ((< (vx added) (- 0 w)) (v+ added (vector2 (* w 3) 0)))
                       ((> (vy added) (+ h h)) (v- added (vector2 0 (* h 3))))
                       ((< (vy added) (- 0 h)) (v+ added (vector2 0 (* h 3))))
                       (else added)))
               (vector2 0 0)
               *time*
               map-background-delta))

(define (spawn-character-pyramid prototype position)
  (define sprite (load-tileset "images/pyramid.png" 388 290))
  (define delta 60)
  (define animations
    (list
     (make-animation-stance
      'idle
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4)))))
  (define animations-hash
    (alist->hash-table
     (map (lambda (ani) (cons (animation-stance-stance ani) ani))
          animations)))
  (make-character 'pyramid
                  position
                  1 ; z
                  'idle
                  animations-hash))
(define (spawn-character-pyramid-s prototype position)
  (define sprite (load-tileset "images/pyramid-s.png" 194 145))
  (define delta 60)
  (define animations
    (list
     (make-animation-stance
      'idle
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4))
      (make-animation sprite 0 delta 5 (list 0 1 2 3 4)))))
  (define animations-hash
    (alist->hash-table
     (map (lambda (ani) (cons (animation-stance-stance ani) ani))
          animations)))
  (make-character 'pyramid-s
                  position
                  1 ; z
                  'idle
                  animations-hash))

(define (spawn-character-birb tilemap prototype position)
  (define sprite (load-tileset "images/birb.png" 16 16))
  (define delta 23)
  (define animations
    (list
     (make-animation-stance
      'normal-walk
      (make-animation sprite 1 delta 3 (list 0 1 2 1))
      (make-animation sprite 0 delta 3 (list 0 1 2 1))
      (make-animation sprite 1 delta 3 (list 0 1 2 1))
      (make-animation sprite 0 delta 3 (list 0 1 2 1)))
     (make-animation-stance
      'idle
      (make-animation sprite 2 120 3 (list 0 1 2 1 0 0 0 0 0 0 0 0 0 0))
      (make-animation sprite 2 120 3 (list 0 1 2 1 0 0 0 0 0 0 0 0 0 0))
      (make-animation sprite 2 120 3 (list 0 1 2 1 0 0 0 0 0 0 0 0 0 0))
      (make-animation sprite 2 120 3 (list 0 1 2 1 0 0 0 0 0 0 0 0 0 0)))))
  (define animations-hash
    (alist->hash-table
     (map (lambda (ani) (cons (animation-stance-stance ani) ani))
          animations)))
  (warn "birb " (tilemap-object-with-name tilemap 'waypoint-birb))
  (character-set-next-position
   (character-set-move-duration
    (make-character 'birb
                    position
                    1                   ; z
                    'idle
                    animations-hash)
    230)
   (rect-position
    (tilemap-object-rect
     (tilemap-object-with-name tilemap 'waypoint-birb)))))

(define (spawn-character tilemap prototype position)
  (cond ((eq? prototype 'pyramid)   (spawn-character-pyramid prototype position))
        ((eq? prototype 'pyramid-s) (spawn-character-pyramid-s prototype position))
        ((eq? prototype 'birb) (if (and (player-has-visited (signal-ref *player*) 'nexus) (random-chance 5))
                                   (spawn-character-birb tilemap prototype position)
                                   #f))
        (else (warn "no character for prototype" prototype) #f)))

(define-signal map-characters '())
(define-signal map-characters
  (signal-let ((tilemap tilemap)
               (time *time*)
               (needs-to-spawn needs-to-spawn-npcs))
              (cond ((and needs-to-spawn tilemap)
                     (let ()
                       (define layer (object-layer-named tilemap 'characters))
                       (define character-objects (or (and layer (tilemap-object-layer-objects layer))
                                                     '()))
                       (signal-set! needs-to-spawn-npcs #f)
                       (filter
                        (lambda (v) (if v #t (begin (warn "no character protoype given" #f))))
                        (map (lambda (object)
                               (define prototype
                                 (string->symbolM
                                  (object-get-property-value object 'prototype)))
                               (define position (rect-position (tilemap-object-rect object)))
                               (and prototype (spawn-character tilemap prototype position)))
                             character-objects))))
                    ((signal-ref map-characters)
                     (filter
                      identity
                      (map (lambda (c) (character-update time c))
                           (signal-ref map-characters))))
                    (else '()))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; CONVERSIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (position->tilecoords pos)
  (vmap inexact->exact
        (vector2 (modulo (floor (/ (vx pos) tile-width)) (signal-ref tiles-per-x))
                 (modulo (floor (/ (vy pos) tile-height)) (signal-ref tiles-per-y)))))

(define (position->tileid pos)
  (tilecoords->tileid
   (position->tilecoords pos)))

(define (tilecoords->position coords)
  (v* coords
      tile-dimensions))

(define (tilecoords->tileid coords)
  (+ (* (vy coords) tiles-per-x)
     (vx coords)))

(define (tileid->coords tileid)
  (vector2 (modulo tileid tiles-per-x)
           (/      tileid tiles-per-y)))

(define (tilecoords->rect coords)
  (make-rect (tilecoords->position coords)
             tile-dimensions))

(define (touches-tile? rect coords)
  (define tr (tilecoords->rect coords))
  (rects-touch? rect tr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; UTILS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (is-on-map rect)
  (rect-inside? (rect-position rect)
                (make-rect 0 0
                           (signal-ref map-width)
                           (signal-ref map-height))))

(define (collides-with-map tilemap rect)
  (define collision-layer (layer-named tilemap 'collision))
  ;; TODO optimize only test tiles in area
  (define y-itr 0)
  (define x-itr 0)
  (define layer-height (if collision-layer
                           (- (length (tilemap-layer-indices collision-layer)) 1)
                           #f))
  (define player-tile-coords (position->tilecoords (rect-position rect)))
  (cond ((not (or (is-on-map rect)
                  (signal-ref map-loops)))
         ;; (warn "outside of map area" rect)
         ;; (rect-inside?
         ;;  (rect-position rect)
         ;;  (rect-scale-center (make-rect 0 0
         ;;                                (signal-ref map-width)
         ;;                                (signal-ref map-height))
         ;;                     1.2))
         #t)
        (collision-layer
         (find (lambda (line)
                 (define y y-itr)
                 (define sly-y y)
                 (define line-result
                   (find identity (filter (lambda (tile)
                                            (define x x-itr)
                                            (set! x-itr (+ x-itr 1))
                                            (if (= tile 1161)
                                                (let ()
                                                  (and (= x (vx player-tile-coords))
                                                       (= sly-y (vy player-tile-coords))))
                                                #f))
                                          line)))
                 (set! x-itr 0)
                 (set! y-itr (+ y-itr 1))
                 line-result)
               (tilemap-layer-indices collision-layer)))
        (tilemap (warn "NO Collision layer on tilemap")
                 #f)
        (else #f)))

;;; main character shit
(define-signal action-on-next-tile #f)
(define-signal locked-action #f)
(define-signal sleep-timer-offset #f)
(define* (change-map use-spawn tmx-path #:optional (fade-duration 23) (fade-image "images/fade.png") (stay-duration 5) (allow-simple-override #f))
  (define changefade
    (make-fade
     fade-to-black
     fade-from-black
     fade-duration
     stay-duration
     fade-image
     allow-simple-override
     (lambda ()
       (define default-room-spawn (symbol->string (signal-ref map-current-name)))
       (warn "CHANGING" default-room-spawn use-spawn)
       (signal-set! map-use-spawn (if (or (not use-spawn)
                                          (string-null? use-spawn))
                                      default-room-spawn
                                      use-spawn))
       (signal-set! tilemap-file tmx-path)
       (signal-set! needs-to-spawn #t)
       (signal-set! needs-to-spawn-npcs #t)
       (signal-set! locked-action #f)
       (signal-set! action-on-next-tile #f)
       (signal-set! sleep-timer-offset #f)
       (signal-set! *player* (player-add-visit (signal-ref *player*) (simplified-map-name tmx-path)))
       (warn "map changed" tmx-path))))
  (fade-activate! changefade))

(define (colliding-logic-object tilemap rect type)
  (define logic-layer (object-layer-named tilemap 'logic))
  (define objects (objects-with-property logic-layer type))
  (define object (find (lambda (object)
                         (rects-touch? (tilemap-object-rect object) rect))
                       objects))
  object)

(define (handle-collision-with-portals tilemap rect)
  (define portal (colliding-logic-object tilemap rect 'portal))
  (define portal-tmx (and portal
                          (object-get-property portal 'portal)
                          (property-value (object-get-property portal 'portal))))
  (if (and portal-tmx (not (string-null? portal-tmx)))
      (let ()
        (define use-spawn (property-value (object-get-property portal 'use-spawn)))
        (define fade-duration (or (object-get-property-value portal 'fade-duration) 23))
        (define fade-image (or (object-get-property-value portal 'fade-image) "images/fade.png"))
        (play-portal-sound portal)
        (change-map use-spawn portal-tmx fade-duration fade-image)
        portal)
      #f))
