(define-module (tiled)
  #:use-module (sxml simple)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-9)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (html)
  #:use-module (utils)
  #:use-module (sly math rect)
  #:use-module (sly render color)
  #:export (load-tmx
            tilemap-tileset
            tilemap-tileset-path
            tilemap-tiles-per-x
            tilemap-tiles-per-y
            tilemap-layers
            tilemap-hidden-layers
            tilemap-object-layers

            tilemap-layer-name
            tilemap-layer-indices
            tilemap-layer-width
            tilemap-layer-height
            tilemap-layer-z
            tilemap-layer-indices

            tilemap-object-layer-name
            tilemap-object-layer-objects

            tilemap-object-id
            tilemap-object-gid
            tilemap-object-shape
            tilemap-object-rect

            unhide-layer
            hide-layer
            layer-named
            object-layer-named
            tilemap-get-map-property
            tilemap-object-with-property
            tilemap-object-with-type
            tilemap-object-with-name
            objects-with-property
            tilemap-property
            object-get-property
            object-get-property-value
            property-name
            property-value))

(define-immutable-record-type <tilemap>
  (make-tilemap tileset tileset-path tiles-per-x tiles-per-y layers hidden-layers object-layers properties)
  tilemap?
  (tileset tilemap-tileset)
  (tileset-path tilemap-tileset-path)
  (tiles-per-x tilemap-tiles-per-x)
  (tiles-per-y tilemap-tiles-per-y)
  (layers tilemap-layers tilemap-set-layers)
  (hidden-layers tilemap-hidden-layers tilemap-set-hidden-layers)
  (object-layers tilemap-object-layers)
  (properties tilemap-properties))

(define-immutable-record-type <tilemap-layer>
  (make-tilemap-layer name width height indices z visible)
  tilemap-layer?
  (name tilemap-layer-name)
  (width tilemap-layer-width)
  (height tilemap-layer-height)
  (indices tilemap-layer-indices)
  (z tilemap-layer-z)
  (visible tilemap-layer-visible))

(define-immutable-record-type <tilemap-object-layer>
  (make-tilemap-object-layer name objects z visible properties)
  tilemap-object-layer?
  (name tilemap-object-layer-name)
  (objects tilemap-object-layer-objects)
  (z tilemap-object-layer-z)
  (visible tilemap-object-layer-visible)
  (properties tilemap-object-layer-properties))

(define-immutable-record-type <tilemap-object>
  (make-tilemap-object name id gid rect shape type properties)
  tilemap-object?
  (name tilemap-object-name)
  (id tilemap-object-id)
  (gid tilemap-object-gid)
  (rect tilemap-object-rect)
  (shape tilemap-object-shape)
  (type tilemap-object-type)
  (properties tilemap-object-properties))

(define* (map-enumerated proc . lsts)
  (define counter 0)
  (apply map (lambda* (. items)
               (define result (apply proc counter items))
               (set! counter (+ 1 counter))
               result)
         lsts))

(define* (load-tmx file #:optional (load-tileset (lambda (path tw th) #f)))
  (define sxml (call-with-input-file file
                 (lambda (port)
                   (xml->sxml port))))
  (define tileset-entry (html-match1 sxml '(tileset)))
  (define tileset-source (html-match-attr1 tileset-entry '(tileset) 'source))
  (define tileset (if tileset-source (read-tileset tileset-source) #f))
  (define tileset-width 40)
  (define tileset-height 30)
  (define tileset-image-path tileset)
  (define layers (map-enumerated (lambda (z layer)
                                   (read-layer layer z tileset-width tileset-height))
                                 (reverse (delete-duplicates (html-match sxml '(layer))))))
  (define map-tag (html-match1 sxml '(map)))
  (define tiles-per-x (string->number (html-match-attr1 map-tag '(map) 'width)))
  (define tiles-per-y (string->number (html-match-attr1 map-tag '(map) 'height)))
  (define tile-width  (string->number (html-match-attr1 map-tag '(map) 'tilewidth)))
  (define tile-height (string->number (html-match-attr1 map-tag '(map) 'tileheight)))
  (define object-layers
    (map-enumerated
     (lambda (z olayer)
       (read-object-layer olayer z (* tiles-per-y tile-height)))
     (reverse (delete-duplicates (html-match sxml '(objectgroup))))))

  (hide-layer (make-tilemap
               (if tileset
                   (load-tileset (string-append "rooms/" tileset) 16 16)
                   (error "no tileset on map" file))
               tileset
               tiles-per-x
               tiles-per-y
               (filter (lambda (l)
                         (tilemap-layer-visible l))
                       layers)
               (filter (lambda (l)
                         (not (tilemap-layer-visible l)))
                       layers)
               object-layers
               (read-properties sxml 'map))
              'collision))

(define (read-tileset file)
  (define sxml (call-with-input-file (string-append "rooms/" file)
                 (lambda (port)
                   (xml->sxml port))))
  (define image (html-match1 sxml '(image)))
  (define source (html-match-attr1 image '(image) 'source))
  source)

(define (read-layer layer z tileset-width tileset-height)
  (define width (string->number (html-match-attr1 layer '(layer) 'width)))
  (define height (string->number (html-match-attr1 layer '(layer) 'height)))
  (define data-tag (html-match1 layer '(data)))
  (define data-encoding (html-match-attr1 data-tag '(data) 'encoding))
  (define indices
    (if (string-ci=? "csv" data-encoding)
        (parse-indices (html-text data-tag) width tileset-width tileset-height)
        (error "tilemap data must be in csv format. It's in:"
               data-encoding)))
  (define visible (not (equal? "0" (html-match-attr1 layer '(layer) 'visible))))

  (make-tilemap-layer
   (string->symbol (html-match-attr1 layer '(layer) 'name))
   width
   height
   indices
   z
   visible))

(define (parse-indices text map-tiles-per-x line-length max-lines)
  (define indices-str (string-split text #\,))
  (define indices (map (lambda (str)
                         (string->number (string-trim-both str)))
                       indices-str))
  (define (to-sly-index index)
    (define orig-y (floor (/ index line-length)))
    (define orig-x (modulo index line-length))
    (define sly-y (- max-lines orig-y 1))
    ;; (warn orig-x orig-y sly-y (* sly-y line-length))
    (if (< sly-y 0)
        (begin (warn "index below 0?" sly-y "tile-index" index
                     "layer dimensions" line-length max-lines
                     "coords-orig" orig-x orig-y
                     "coords-sly" orig-x sly-y)
               (warn "you might be using multiple tiles-sets. Only 1 set per map is supported. Remove the others. Or you hit some bad bug connected to the last column not rendering.")
               1161)
        (if (= index 0)
            (+ (* (- max-lines 1) line-length) 0)
            (+ (* sly-y line-length) (max 0 (- orig-x 1)))))
    )
  (define lined-indices (fold (lambda (index acc)
                                (define counter (car acc))
                                (define cur-list (cons (to-sly-index index) (cadr acc)))
                                (define all-lists (cddr acc))
                                (if (= counter (- map-tiles-per-x 1))
                                    (begin
                                      (set! counter 0)
                                      (set! all-lists (cons (reverse cur-list) all-lists))
                                      (set! cur-list '()))
                                    (set! counter (+ 1 counter)))
                                (cons* counter cur-list all-lists))
                              '(0 () . ())
                              indices))
  ;; (warn indices)
  ;; (warn (reverse (cddr lined-indices)))
  (cddr lined-indices))

(define (read-object-layer layer z map-height-px)
  (define visible (not (equal? "0" (html-match-attr1 layer '(objectgroup) 'visible))))
  (define objects (map (lambda (object)
                         (read-object object map-height-px))
                       (delete-duplicates (html-match layer '(object)))))
  (make-tilemap-object-layer
   (string->symbol (html-match-attr1 layer '(objectgroup) 'name))
   objects
   z
   visible
   (read-properties layer 'layer)))

(define (read-object object map-height-px)
  (define (to-sly-y gid h y)
    (if gid
        (and y (- map-height-px y)) ; graphic stamps have bl origin
        (and y (- map-height-px y h))))
  (define gid (string->numberM (html-match-attr1 object '(object) 'gid)))
  (define w (or (string->numberM (html-match-attr1 object '(object) 'width)) 0))
  (define h (or (string->numberM (html-match-attr1 object '(object) 'height)) 0))
  (define x (string->numberM (html-match-attr1 object '(object) 'x)))
  (define y (to-sly-y gid h (string->numberM (html-match-attr1 object '(object) 'y))))
  (define (read-polyline)
    (define points-str (html-match-attr1 object '(polyline) 'points))
    (define points-num (and points-str
                            (map (lambda (str)
                                   (string->numberM (string-trim-both str)))
                                 (string-split points-str #\,))))
    (and points-num `(polyline . ,points-num)))
  ;; TODO almost the same code as polyline
  (define (read-polygon)
    (define points-str (html-match-attr1 object '(polygon) 'points))
    (define points-num (and points-str
                            (map (lambda (str)
                                   (string->numberM (string-trim-both str)))
                                 (string-split points-str #\,))))
    (and points-num `(polygon . ,points-num)))
  (define (read-ellipse) (if (html-match1 object '(ellipse))
                             '(ellipse . nothing)
                             #f))
  (define (read-text)
    ;; TODO    <text fontfamily="Nimbus Mono L" wrap="1" color="#ff00ff" bold="1" halign="right" valign="center">Hallo Welt</text>
    (define text-tag (html-match1 object '(text)))
    (define text (html-text text-tag))
    `(text . ,text))
  (define shape (or (read-ellipse)
                    (read-polygon)
                    (read-polyline)
                    (read-text)
                    '(rect . nothing)))

  (make-tilemap-object
   (string->symbolM (html-match-attr1 object '(object) 'name))
   (string->numberM (html-match-attr1 object '(object) 'id))
   gid
   (make-rect x y w h)
   shape
   (string->symbolM (html-match-attr1 object '(object) 'type))
   (read-properties object 'object)))

(define (read-properties element parent-tag-name)
  (define properties-parent (html-match element `(,parent-tag-name (properties (property)))))
  (define properties (delete-duplicates (html-match properties-parent '(property))))
  (define (read-property property)
    (define type (or (html-match-attr1 property '(property) 'type) "string"))
    (define name (html-match-attr1 property '(property) 'name))
    (define value-str (html-match-attr1 property '(property) 'value))
    (cons
     (string->symbol name)
     (cond ((or (string=? type "int")
                (string=? type "float"))
            (string->numberM value-str))
           ((string=? type "bool") (string=? "true" value-str))
           ((string=? type "color")
            ;; (let ((match (string-match "^#([0-9abcdef][0-9abcdef])([0-9abcdef][0-9abcdef])([0-9abcdef][0-9abcdef])([0-9abcdef][0-9abcdef])$"
            ;;                             (string-downcase value-str)))
            ;;       (hexstr->int (lambda (str)
            ;;                      ;; TODO implement this
            ;;                      )))
            ;;   (if match
            ;;       (let ((r (/ (hexstr->int (match:substring match 1)) 255))
            ;;             (g (/ (hexstr->int (match:substring match 2)) 255))
            ;;             (b (/ (hexstr->int (match:substring match 3)) 255))
            ;;             (a (/ (hexstr->int (match:substring match 4)) 255)))
            ;;         (make-color r g b a))
            ;;       (make-color 1 1 1 1))
            value-str)
           ((and (string=? type "file") value-str)
            (if (string-null? value-str)
                #f
                (string-append "rooms/" value-str)))
           (else value-str))))
  (map read-property properties))
(define property-name car-safe)
(define property-value cdr-safe)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; utilities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define* (layer-named tilemap name #:optional (layers-getter tilemap-hidden-layers))
  (define layers (layers-getter tilemap))
  (find (lambda (layer)
          (eq? (tilemap-layer-name layer) name))
        layers))

(define (object-layer-named tilemap name)
  (define layers (tilemap-object-layers tilemap))
  (find (lambda (layer)
          (eq? (tilemap-object-layer-name layer) name))
        layers))

(define* (tilemap-get-map-property tilemap name #:optional
                                   (value 'qtiled-no-value-asked))
  (find (lambda (property)
          (property-matches property name value))
        (tilemap-properties tilemap)))

(define* (tilemap-object-with-property tilemap name #:optional
                                       (value 'qtiled-no-value-asked)
                                       (layers (tilemap-object-layers tilemap)))
  (find-compute
   (lambda (olayer)
     (find (lambda (object)
             (object-get-property object name value))
           (tilemap-object-layer-objects olayer)))
   layers))

(define* (tilemap-object-with-type tilemap type)
  (find-compute
   (lambda (olayer)
     (find (lambda (object)
             (eq? (tilemap-object-type object) type))
           (tilemap-object-layer-objects olayer)))
   (tilemap-object-layers tilemap)))

(define* (tilemap-object-with-name tilemap name)
  (find-compute
   (lambda (olayer)
     (find (lambda (object)
             (eq? (tilemap-object-name object) name))
           (tilemap-object-layer-objects olayer)))
   (tilemap-object-layers tilemap)))

(define* (objects-with-property olayer name #:optional
                                (value 'qtiled-no-value-asked))
  (filter (lambda (object)
            (object-get-property object name value))
          (tilemap-object-layer-objects olayer)))

(define* (object-get-property object name #:optional (value 'qtiled-no-value-asked))
  (find (lambda (property)
          (property-matches property name value))
        (tilemap-object-properties object)))

(define (object-get-property-value object name)
  "returns #f if there is no property, so don't confuse it with booleans"
  (define property (and object (object-get-property object name)))
  (and property (property-value property)))


(define (property-matches property name value)
  (and (eq? (property-name property) name)
       (or (eq? value 'qtiled-no-value-asked)
           (equal? (property-value property) value))))

(define* (tilemap-property tilemap name #:optional
                           (value 'qtiled-no-value-asked))
  (define result (find (lambda (property)
                         (property-matches property name value))
                       (tilemap-properties tilemap)))
  (and result (property-value result)))

(define (unhide-layer tilemap name)
  (define layer (layer-named tilemap name))
  (define new-visible (and layer
                           (sort (cons layer (tilemap-layers tilemap))
                                 (lambda (a b)
                                   (< (tilemap-layer-z a) (tilemap-layer-z b))))))
  (define new-hidden (delete layer (tilemap-hidden-layers tilemap)))
  (if (and layer new-visible new-hidden)
      (tilemap-set-layers (tilemap-set-hidden-layers tilemap new-hidden)
                          new-visible)
      (begin
        ;; (warn "can not unhide layer " name "it's not hidden")
        tilemap)))

(define (hide-layer tilemap name)
  (define layer (layer-named tilemap name tilemap-layers))
  (define new-hidden (and layer
                          (sort (cons layer (tilemap-hidden-layers tilemap))
                                (lambda (a b)
                                  (< (tilemap-layer-z a) (tilemap-layer-z b))))))
  (define new-visible (delete layer (tilemap-layers tilemap)))
  (if (and layer new-visible new-hidden)
      (tilemap-set-layers (tilemap-set-hidden-layers tilemap new-hidden)
                          new-visible)
      (begin
        ;; (warn "can not hide layer " name "it's not hidden")
        tilemap)))
