(define-module (character)
  #:use-module (srfi srfi-9)            ;record
  #:use-module (srfi srfi-9 gnu)        ;immutable record
  #:use-module (sly)
  #:use-module (utils)
  #:use-module (sly math)
  #:use-module (sly math rect)
  #:use-module ((sly math transform) #:prefix t:)
  #:use-module (sly math tween)
  #:use-module (sly math vector)
  #:use-module (sly render tileset)
  #:use-module ((srfi srfi-43)  #:prefix stdvector:) ; vector
  ;; #:use-module (adrian utils)
  ;; #:use-module (adrian video globals)
  #:export (load-character
            make-character
            character-id
            character-position character-set-position
            character-next-position character-set-next-position
            character-z
            character-direction character-set-direction
            character-stance character-set-stance
            character-move-duration character-set-move-duration

            character-update
            character-is-moving?
            character-move
            character-is-in-move-animation?
            character-finalize-move-animation
            character-render
            make-animation
            make-animation-stance

            animation-stance-stance

            character-collision-rect))

(define-immutable-record-type <character>
  (%make-character id position next-position move-timer movetween move-animation-timer
                   move-duration z stance direction animations)
  character?
  (id character-id)
  (position character-position character-set-position)
  (next-position character-next-position %character-set-next-position)
  (move-timer character-move-timer character-set-move-timer)
  (movetween character-movetween character-set-movetween)
  (move-animation-timer character-move-animation-timer character-set-move-animation-timer)
  (move-duration character-move-duration character-set-move-duration)
  (z character-z)
  (stance character-stance character-set-stance)
  (direction character-direction character-set-direction)
  (animations character-animations))

(define-immutable-record-type <animation>
  (%make-animation cycle tween)
  animation?
  (cycle animation-cycle)
  (tween animation-tween))

(define-immutable-record-type <animation-stance>
  (make-animation-stance stance left right down up)
  animation-stance?
  (stance animation-stance-stance)
  (left animation-stance-left)
  (right animation-stance-right)
  (down animation-stance-down)
  (up animation-stance-up))

(define (make-animation tiles y-row delta-time per-row frames)
  (define animation-cycle (make-walk-cycle tiles y-row per-row frames))
  (define animation-tween (make-frame-tween animation-cycle delta-time))
  (%make-animation animation-cycle animation-tween))

(define default-move-duration 17)
(define (make-character id position z stance animation-data)
  (define step-time 90)
  (define pos position ;; (make-position-signal next-pos-sig step-time-sig)
    )
  (%make-character
   id
   pos
   #f
   #f
   #f
   #f
   default-move-duration
   z
   stance
   'down
   animation-data))

(define move-animation-duration 60)
(define (character-set-next-position c position)
  (define move-duration (character-move-duration c))
  (define movetween (tween vlerp ease-linear
                           (character-position c)
                           position
                           move-duration))
  (define tween-c (character-set-movetween c movetween))
  (define timer-c (character-set-move-timer tween-c (signal-timer)))
  (character-set-move-animation-timer
   (character-set-stance
    (%character-set-next-position timer-c position)
    'normal-walk)
   (or (character-move-animation-timer c) (signal-timer))))
(define (character-move c time)
  (if (character-is-moving? c)
      (let ()
        (define move-duration (character-move-duration c))
        (define movetween (character-movetween c))
        (define timer (character-move-timer c))
        (define time (min move-duration (signal-ref timer)))
        (define moved-c (character-set-position c (movetween time)))
        (if (= time move-duration)
            (%character-set-next-position
             (character-set-move-timer
              (character-set-movetween moved-c #f)
              #f)
             #f)
            moved-c))
      #f))
(define (character-is-in-move-animation? c)
  (and c (character-move-animation-timer c)))
(define (character-finalize-move-animation c)
  (define tween (animation-tween (character-animation c)))
  (define time (signal-ref (character-move-animation-timer c)))
  (define frame-id (tween time))
  ;; is idle frame or over?
  (if  (or (= 1 frame-id)
           (= 3 frame-id)
           (>= (signal-ref (character-move-animation-timer c))
               move-animation-duration))
       (character-set-stance
        (character-set-move-animation-timer c #f)
        'idle)
       c))

(define (make-position-signal next-tile-sig step-time-sig)
  (signal-fold
   (lambda (target-pos step-time t current-pos)
     ;; (define fake-target (vector2 (random 200) (random 200)))
     ;; (warn "POS" fake-target)
     ;; fake-target
     current-pos
     ;; (tween vlerp (compose ease-linear ease-loop)
     ;;        current-pos fake-target 90)
     )
   (vector2 0 0)
   next-tile-sig step-time-sig *time*))

(define (character-animation c)
  (define animation-data (character-animations c))
  (define direction (character-direction c))
  (define stance (character-stance c))
  (define animation-stance (hash-ref animation-data stance))
  (if animation-stance
      (cond ((eq? direction 'left)  (animation-stance-left  animation-stance))
            ((eq? direction 'right) (animation-stance-right animation-stance))
            ((eq? direction 'up)    (animation-stance-up    animation-stance))
            ((eq? direction 'down)  (animation-stance-down  animation-stance))
            (else (warn "UNKNOWN WALK DIRECTION for character" direction)
                  (animation-stance-down  animation-stance)))
      (begin
        (warn "UNKNOWN ANIMATION STANCE for character" stance)
        #f)))

(define (make-walk-cycle tiles y-row animations-per-row frames)
  (define start-id (* y-row animations-per-row))
  (stdvector:list->vector
   (map (lambda (id)
          (make-sprite (tileset-ref tiles id)))
        (map (lambda (fid) (+ start-id fid)) frames))))

(define (render-animation c given-time a)
  (define cycle (animation-cycle a))
  (define tween (animation-tween a))
  (define time (if (character-is-in-move-animation? c)
                   (signal-ref (character-move-animation-timer c))
                   given-time))
  (let* ((frame (vector-ref cycle (tween time))))
    (render-sprite frame)))

(define (make-frame-tween walk-cycle delta)
  (if (stdvector:vector-empty? walk-cycle)
      (const 0)
      (let* ((frame-count (vector-length walk-cycle))
             (frame-rate delta))
        (tween (compose floor lerp) (compose ease-linear ease-loop)
               0 frame-count delta))))

(define (character-update time c)
  (cond
   ((character-is-moving? c)
    ;; (play-step-sound tm (character-next-position c))
    (let ((new-c (character-move c time)))
      ;; (when (not (character-is-moving? new-mc))
      ;;   (handle-collision-with-portals tm (character-collision-rect
      ;;                                      (character-position new-mc))))
      new-c))
   ((character-is-in-move-animation? c)
    (character-finalize-move-animation c))
   (else c)))

(define (character-render time c)
  (define pos (character-position c))
  (define animation (character-animation c))
  (if (and pos animation)
      (move (v* pos *scale*) (render-animation c time animation))
      render-nothing))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (character-is-moving? c)
  (and c
       (character-movetween c)
       (character-move-timer c)
       (character-next-position c)))

(define (load-character sprite-path)
  (define sprite (load-tileset sprite-path 24 32))
  (define id 0)
  (define delta move-animation-duration)
  (define walk-frames (list 0 1 2 1))
  (define idle-frames (list 1))
  (define per-row 3)
  (define animations
    (list
     (make-animation-stance
      'normal-walk
      ;; original yn sprites y-rows are: up, right, down, left
      ;; but this engine is fugged, so it uses bottom-left as origin
      ;; actually: left, down, right, up
      ;; we use left, right, down, up
      (make-animation sprite (+ 4 0) delta per-row walk-frames)
      (make-animation sprite (+ 4 2) delta per-row walk-frames)
      (make-animation sprite (+ 4 1) delta per-row walk-frames)
      (make-animation sprite (+ 4 3) delta per-row walk-frames))
     (make-animation-stance
      'idle
      (make-animation sprite (+ 4 0) delta per-row idle-frames)
      (make-animation sprite (+ 4 2) delta per-row idle-frames)
      (make-animation sprite (+ 4 1) delta per-row idle-frames)
      (make-animation sprite (+ 4 3) delta per-row idle-frames))
     (make-animation-stance
      'sleep
      (make-animation sprite (+ 0 0) 1 per-row idle-frames)
      (make-animation sprite (+ 0 0) 1 per-row idle-frames)
      (make-animation sprite (+ 0 0) 1 per-row idle-frames)
      (make-animation sprite (+ 0 0) 1 per-row idle-frames))))
  (define animations-hash
    (alist->hash-table
     (map (lambda (ani) (cons (animation-stance-stance ani) ani))
          animations)))

  (make-character
   id
   (vector2 500 250)
   1 ; z
   'idle
   animations-hash))


(define player-collision-width 12)
(define player-collision-height 12)
;;; TODO match
(define (character-collision-rect position-or-character)
  (if (character? position-or-character)
      (character-collision-rect (character-position position-or-character))
      (make-rect (- (vx position-or-character) 8)
                 (- (vy position-or-character) 12)
                 player-collision-width
                 player-collision-height)))
