//==========================================================================
// EliMZ_MapTransitions.js
//==========================================================================

/*:
@target MZ
@base EliMZ_Book

@plugindesc ♦1.1.0♦ Adds new map transitions!
@author Hakuen Studio
@url https://hakuenstudio.itch.io/rate?source=game

@help
★★★★★ Rate the plugin! Please, is very important to me ^^

● Terms of Use
https://www.hakuenstudio.com/terms-of-use-5-0-0

============================================================================
Features
============================================================================

● Adds different Map transitions via map note tag or variable!

- Fade
- Cross Fade
- Zoom
- Zoom Cross Fade
- Pixelated
- Split Screen Horizontal
- Split Screen Vertical
- Split Screen Horz and Vert
- Move Screen Left
- Move Screen Right
- Move Screen Top
- Move Screen Bottom

============================================================================
How to use
============================================================================

https://docs.google.com/document/d/1LYqdNj6bUjMEaMFIvxO-NeI0N1i4Nl1FzBgvfm89t7w/edit?usp=sharing

============================================================================

@param transitions
@text Transition List
@type struct<transitionSt>[]
@desc
@default ["{\"id\":\"StandardFade\",\"type\":\"Fade\",\"--Transition Config--\":\"\",\"fade\":\"{\\\"easing\\\":\\\"linear\\\",\\\"duration\\\":\\\"24\\\"}\",\"crossFade\":\"\",\"zoom\":\"\",\"zoomCrossFade\":\"\",\"pixel\":\"\"}","{\"id\":\"CrossFade\",\"type\":\"Cross Fade\",\"--Transition Config--\":\"\",\"fade\":\"\",\"crossFade\":\"{\\\"easing\\\":\\\"linear\\\",\\\"duration\\\":\\\"30\\\"}\",\"zoom\":\"\",\"zoomCrossFade\":\"\",\"pixel\":\"\"}","{\"id\":\"ZoomCrossFade\",\"type\":\"Zoom Cross Fade\",\"--Transition Config--\":\"\",\"fade\":\"\",\"crossFade\":\"\",\"zoom\":\"\",\"zoomCrossFade\":\"{\\\"zoomTarget\\\":\\\"32\\\",\\\"zoomDuration\\\":\\\"60\\\",\\\"zoomEasing\\\":\\\"linear\\\",\\\"fadeDuration\\\":\\\"30\\\"}\",\"pixel\":\"\"}","{\"id\":\"StandardZoom\",\"type\":\"Zoom\",\"--Transition Config--\":\"\",\"fade\":\"\",\"crossFade\":\"\",\"zoom\":\"{\\\"zoomTarget\\\":\\\"32\\\",\\\"zoomDuration\\\":\\\"60\\\",\\\"zoomEasing\\\":\\\"linear\\\",\\\"fadeDuration\\\":\\\"30\\\"}\",\"zoomCrossFade\":\"\",\"pixel\":\"\"}","{\"id\":\"Pixelated\",\"type\":\"Pixel\",\"--Transition Config--\":\"\",\"fade\":\"\",\"crossFade\":\"\",\"zoom\":\"\",\"zoomCrossFade\":\"\",\"pixel\":\"{\\\"pixelTarget\\\":\\\"32\\\",\\\"pixelDuration\\\":\\\"60\\\",\\\"pixelEasing\\\":\\\"linear\\\",\\\"fadeDuration\\\":\\\"30\\\"}\"}"]

@param defSceneTransition
@text Default Transition
@type text
@desc
@default 0

@param varInTransition
@text Variable Transition
@type variable
@desc This variable will hold the transition id.
@default 0

*/

/* ------------------------------- TRANSITION ------------------------------- */
{

/*~struct~transitionSt:

@param id
@text ID
@type text
@desc Cannot be a number and cannot have spaces. Not case sensitive.
@default MyTransition.

@param type
@text Type
@type select
@option Fade
@option Cross Fade
@option Zoom
@option Zoom Cross Fade
@option Pixel
@option Split Screen Horizontal
@option Split Screen Vertical
@option Split Screen Horz and Vert
@option Move Screen Left
@option Move Screen Right
@option Move Screen Top
@option Move Screen Bottom
@desc
@default Fade

@param --Transition Config--

@param fade
@text Fade
@type struct<fadeSt>
@desc
@default

@param crossFade
@text Cross Fade
@type struct<crossFadeSt>
@desc
@default

@param zoom
@text Zoom
@type struct<zoomFadeSt>
@desc
@default

@param zoomCrossFade
@text Zoom Cross Fade
@type struct<zoomCrossFadeSt>
@desc
@default

@param pixel
@text Pixel
@type struct<pixelSt>
@desc
@default

@param splitHorizontal
@text Split Screen Horizontal
@type struct<fadeSt>
@desc
@default

@param splitVertical
@text Split Screen Vertical
@type struct<fadeSt>
@desc
@default

@param splitHorzAndVert
@text Split Screen Horz and Vert
@type struct<fadeSt>
@desc
@default

@param moveScreenLeft
@text Move Screen Left
@type struct<fadeSt>
@desc
@default

@param moveScreenRight
@text Move Screen Right
@type struct<fadeSt>
@desc
@default

@param moveScreenTop
@text Move Screen Top
@type struct<fadeSt>
@desc
@default

@param moveScreenBottom
@text Move Screen Bottom
@type struct<fadeSt>
@desc
@default

*/

}

/* ---------------------------------- FADE ---------------------------------- */
{

/*~struct~fadeSt:

    @param easing
    @text Easing
    @type combo
    @desc
    @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
    @default linear

    @param duration
    @text Duration
    @type text
    @desc A duration in frames.
    @default 30

*/

}

/* ------------------------------- CROSS FADE ------------------------------- */
{

/*~struct~crossFadeSt:

    @param easing
    @text Easing
    @type combo
    @desc
    @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
    @default linear

    @param duration
    @text Duration
    @type text
    @desc A duration in frames.
    @default 30

*/

}

/* ----------------------------- ZOOM CROSS FADE ---------------------------- */
{

/*~struct~zoomCrossFadeSt:

    @param zoomTarget
    @text Zoom Target Value
    @type text
    @desc The target zoom value
    @default 32

    @param zoomDuration
    @text Zoom Duration
    @type text
    @desc A duration in frames.
    @default 60

    @param zoomEasing
    @text Easing
    @type combo
    @desc
    @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
    @default linear

    @param fadeDuration
    @text Fade Duration Effect
    @type text
    @desc A duration in frames.
    @default 30

*/

}

/* -------------------------------- ZOOM FADE ------------------------------- */
{

/*~struct~zoomFadeSt:

    @param zoomTarget
    @text Zoom Target Value
    @type text
    @desc The target zoom value
    @default 32

    @param zoomDuration
    @text Zoom Duration
    @type text
    @desc A duration in frames.
    @default 60

    @param zoomEasing
    @text Easing
    @type combo
    @desc
    @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
    @default linear

    @param fadeDuration
    @text Fade Duration
    @type text
    @desc A duration in frames.
    @default 30

*/

}

/* ---------------------------------- PIXEL --------------------------------- */
{

/*~struct~pixelSt:

    @param pixelTarget
    @text Pixel Target Value
    @type text
    @desc The target pixel value
    @default 32

    @param pixelDuration
    @text Pixel Duration
    @type text
    @desc A duration in frames.
    @default 60

    @param pixelEasing
    @text Easing
    @type combo
    @desc
    @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
    @default linear

    @param fadeDuration
    @text Fade Duration
    @type text
    @desc A duration in frames.
    @default 32

*/

}


"use strict"

var Eli = Eli || {}
var Imported = Imported || {}
Imported.Eli_SceneTransitions = true

/* ========================================================================== */
/*                                   PLUGIN                                   */
/* ========================================================================== */
Eli.MapTransitions = {

    fadeOutBitmap: null,
    fadeInBitmap: null,
    outType: "",
    inType: "",

    Parameters: class Parameters{

        constructor(parameters){
            this.transitionList = this.parseTransitions(JSON.parse(parameters.transitions))
            this.sceneDefaultTransition = this.findDefaults(parameters.defSceneTransition)
            this.varInTransition = Number(parameters.varInTransition) || 0
            this.varOutTransition = Number(parameters.varOutTransition) || 0
        }

        parseTransitions(parameters){
            const transitionList = []

            for(const param of parameters){
                const transition = JSON.parse(param)
                const type = this.findType(transition.type)
                const data = transition[type] || "{}"

                transitionList.push({
                    id: transition.id.toLowerCase().replaceAll(" ", "") || null,
                    type: type,
                    data: this[`parseTransition_${type}`](JSON.parse(data)),
                })
            }

            return transitionList
        }

        findType(type){
            return {
                "Fade": "fade",
                "Cross Fade": "crossFade",
                "Zoom Cross Fade": "zoomCrossFade",
                "Zoom": "zoom",
                "Pixel": "pixel",
                "Split Screen Horizontal": "splitHorizontal",
                "Split Screen Vertical": "splitVertical",
                "Split Screen Horz and Vert": "splitHorzAndVert",
                "Move Screen Left": "moveScreenLeft",
                "Move Screen Right": "moveScreenRight",
                "Move Screen Top": "moveScreenTop",
                "Move Screen Bottom": "moveScreenBottom",
                "None": "none",
            }[type]
        }

        parseTransition_fade(parameter){
            return {
                easing: parameter.easing,
                duration: Number(parameter.duration)
            }
        }

        parseTransition_crossFade(parameter){
            return {
                easing: parameter.easing,
                duration: Number(parameter.duration)
            } 
        }

        parseTransition_zoomCrossFade(parameter){
            return {
                zoomTarget: Number(parameter.zoomTarget),
                zoomDuration: Number(parameter.zoomDuration),
                zoomEasing: parameter.zoomEasing,
                fadeDuration: Number(parameter.fadeDuration),
            }
        }

        parseTransition_zoom(parameter){
            return {
                zoomTarget: Number(parameter.zoomTarget),
                zoomDuration: Number(parameter.zoomDuration),
                zoomEasing: parameter.zoomEasing,
                fadeDuration: Number(parameter.fadeDuration),
            }
        }

        parseTransition_pixel(parameter){
            return {
                pixelTarget: Number(parameter.pixelTarget),
                pixelDuration: Number(parameter.pixelDuration),
                pixelEasing: parameter.pixelEasing,
                fadeDuration: Number(parameter.fadeDuration),
            }  
        }

        parseTransition_splitHorizontal(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_splitVertical(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_splitHorzAndVert(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_moveScreenLeft(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_moveScreenRight(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_moveScreenTop(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_moveScreenBottom(parameter){
            return this.parseTransition_fade(parameter)
        }

        parseTransition_none(parameter){
            return {}  
        }

        findDefaults(parameter){
            return this.transitionList[parameter] || this.transitionList.find(item => item.id === parameter.toLowerCase().replaceAll(" ", "")) || this.transitionList[0]
        }

    },

    Container: class extends PIXI.Container{

        setInSprite(sprite){
            this.inSprite = sprite
            this.inSprite.visible = false
            this.addChild(this.inSprite)
        }

        setOutSprite(sprite){
            this.outSprite = sprite
            this.addChild(this.outSprite)
        }

        isRunning(){
            return this.children.some(spr => spr.isRunning())
        }

        update(){
            if(this.outSprite){
                this.outSprite.update()

                if(this.inSprite && !this.outSprite.isRunning()){
                    this.outSprite.visible = false
                    this.inSprite.visible = true
                    this.inSprite.update()

                    if(!this.inSprite.isRunning()){
                        this.outSprite.destroy()
                        this.inSprite.destroy()
                        this.outSprite = null
                        this.inSprite = null
                        SceneManager._scene.onTransferEnd()
                    }
                }

            }else if(this.inSprite){
                this.inSprite.visible = true
                this.inSprite.update()

                if(!this.inSprite.isRunning()){
                    this.inSprite.destroy()
                    this.outSprite = null
                    this.inSprite = null
                    SceneManager._scene.onTransferEnd()
                }
            }
        }
    },

    Sprite_Dummy: class extends Sprite{

        initialize(bitmap){
            super.initialize(new Bitmap())
        }

        isRunning(){
            return false
        }
    },

    Sprite_Fade: class extends Sprite{

        initialize(parameters, bitmap, transOut){
            this.initMembers(parameters)
   
            if(transOut){
                super.initialize(bitmap)
                this.createFadeSprite(this.createBlackBitmap())
            }else{
                super.initialize(this.createBlackBitmap())
            }

            this.initAnime(transOut)
        }

        initMembers(parameters){
            this.parameters = parameters
            this.alphaAnime = new Eli.AnimeGroup([], {direction: "normal", paused: false})
        }

        createFadeSprite(blackBitmap){
            this.fadeSprite = new Sprite(blackBitmap)
            this.fadeSprite.alpha = 0
            this.addChild(this.fadeSprite)
        }

        createBlackBitmap(){
            const bitmap = new Bitmap(Graphics.width, Graphics.height)
            bitmap.fillAll("black")

            return bitmap
        }

        initAnime(transOut){
            const defaultData = Eli.AnimeManager.createDefaultData()
            const props = this.createAnimeProps(transOut)
            const subject = transOut ? this.fadeSprite : this
            const animations = Eli.AnimeManager.createAnimations(subject, props, defaultData)

            this.alphaAnime.setAnimations(animations)
            this.alphaAnime.play("normal")
        }

        createAnimeProps(transOut){
            const value = transOut ? [0, 1] : [1, 0]
            return {
                alpha: {value: value, easing: this.parameters.easing, duration: this.parameters.duration},
            }
        }

        isRunning(){
            return this.alphaAnime.isRunning()
        }

        update(){
            this.alphaAnime.update()
        }
    },

    Sprite_CrossFade: class extends Sprite{

        initialize(parameters, bitmap){
            super.initialize(bitmap || new Bitmap())
            this.parameters = parameters
            this.alpha = 1
            this.alphaAnime = new Eli.AnimeTiny(this, "alpha", 0, this.parameters.duration, this.parameters.easing, "normal", 0)
        }

        isRunning(){
            return this.alphaAnime.isRunning()
        }

        update(){
            this.alphaAnime.update()
        }
    },

    Sprite_ZoomCrossFade: class extends Sprite{
        initialize(parameters, bitmap){
            this.parameters = parameters
            super.initialize(bitmap || new Bitmap())
            this.initMembers()
            this.initAnime()
            this.updatePosition()
        }

        initMembers(){
            this.alpha = 1
            this.scaleValue = 1
            this.anchor.set(0.5, 0.5)
            this.animeGroup = new Eli.AnimeGroup([], {direction: "normal", paused: false})
        }

        initAnime(){
            const defaultData = Eli.AnimeManager.createDefaultData()
            const props = this.createAnimeProps()
            const animations = Eli.AnimeManager.createAnimations(this, props, defaultData)
            this.animeGroup.setAnimations(animations)
            this.animeGroup.play("normal")
        }

        createAnimeProps(){
            return {
                scaleValue: {value: [1, this.parameters.zoomTarget], easing: this.parameters.zoomEasing, duration: this.parameters.zoomDuration},
                alpha: {value: [1, 0], easing: "linear", duration: this.parameters.fadeDuration, startDelay: this.parameters.zoomDuration},
            }
        }

        isRunning(){
            return this.animeGroup.isRunning()
        }

        updatePosition(){
            this.x = Graphics.width / 2
            this.y = Graphics.height / 2
        }

        updateScale(){
            this.scale.set(this.scaleValue, this.scaleValue)
        }

        update(){
            this.animeGroup.update()
            this.updatePosition()
            this.updateScale()
        }
    },

    Sprite_Zoom: class extends Sprite{
        initialize(parameters, bitmap){
            this.parameters = parameters
            super.initialize(bitmap)
            this.initMembers()
            this.initAnime()
            this.updatePosition()
        }

        initMembers(){
            this.alpha = 1
            this.scaleValue = 1
            this.anchor.set(0.5, 0.5)
            this.scaleAnime = null
            this.alphaAnime = null
        }

        initAnime(){
            const defaultData = Eli.AnimeManager.createDefaultData()
            const props = this.createAnimeProps()
            const animations = Eli.AnimeManager.createAnimations(this, props, defaultData)

            this.scaleAnime = animations[0]
            this.alphaAnime = animations[1]
            this.alphaAnime.pause()

            this.scaleAnime.data.onComplete = () => {
                this.bitmap = new Bitmap(Graphics.width, Graphics.height)
                this.bitmap.fillAll("black")
                this.bitmap.addLoadListener(() => {
                    this.alphaAnime.play("normal")
                })
                
            }
        }

        createAnimeProps(){
            return {
                scaleValue: {value: [1, this.parameters.zoomTarget], easing: this.parameters.zoomEasing, duration: this.parameters.zoomDuration},
                alpha: {value: [1, 0], easing: "linear", duration: this.parameters.fadeDuration},
            }
        }

        isRunning(){
            return this.scaleAnime.isRunning() || this.alphaAnime.isRunning()
        }

        updatePosition(){
            this.x = Graphics.width / 2
            this.y = Graphics.height / 2
        }

        updateScale(){
            this.scale.set(this.scaleValue, this.scaleValue)
        }

        update(){
            this.scaleAnime.update()
            this.alphaAnime.update()
            this.updatePosition()
            this.updateScale()
        }
    },

    Sprite_Pixel: class extends Sprite{

        initialize(parameters, bitmap){
            this.parameters = parameters
            super.initialize(bitmap || new Bitmap())
            this.initMembers()
            this.initAnime()
        }

        initMembers(){
            this.pixelSize = 0
            this.pixelFilter = new PIXI.filters.PixelateFilter(1)
            this.filters = [this.pixelFilter]
        }

        initAnime(){
            const defaultData = Eli.AnimeManager.createDefaultData()
            const props = this.createAnimeProps()
            const animations = Eli.AnimeManager.createAnimations(this, props, defaultData)

            this.pixelSizeAnime = animations[0]
            this.alphaAnime = animations[1]

            this.pixelSizeAnime.data.onComplete = () => {
                this.bitmap = new Bitmap(Graphics.width, Graphics.height)
                this.bitmap.fillAll("black")
            }
        }

        createAnimeProps(){
            return {
                pixelSize: {value: [1, this.parameters.pixelTarget], easing: this.parameters.pixelEasing, duration: this.parameters.pixelDuration},
                alpha: {value: [1, 0], easing: "linear", duration: this.parameters.fadeDuration, startDelay:this.parameters.pixelDuration},
            }
        }

        isRunning(){
            return this.pixelSizeAnime.isRunning() || this.alphaAnime.isRunning()
        }

        update(){
            this.pixelSizeAnime.update()
            this.alphaAnime.update()
            this.pixelFilter.size = [this.pixelSize, this.pixelSize]
        }
    },

    Sprite_SplitTransition: class Sprite_SplitTransition extends Sprite{

        initialize(parameters, bitmap){
            super.initialize()
            this.createFadeSprite()
            this.parameters = parameters.data
            this.spriteAnimes = []

            if(parameters.type === "splitHorizontal"){
                this.createSplitHorizontal(bitmap)

            }else if(parameters.type === "splitVertical"){
                this.createSplitVertical(bitmap)

            }else if(parameters.type === "splitHorzAndVert"){
                this.createSplitHorzAndVert(bitmap)
            }
        }

        createFadeSprite(){
            this.fadeSprite = new Eli.MapTransitions.Sprite_Fade(Eli.MapTransitions.getParam().sceneDefaultTransition.data, Eli.MapTransitions.fadeInBitmap, false)
            this.fadeSprite.alphaAnime.pause()
            this.addChild(this.fadeSprite)
        }

        createSplitHorizontal(bitmap){
            this.leftSprite = new Sprite(bitmap)
            this.leftSprite.setFrame(0, 0, bitmap.width/2, bitmap.height)
            this.rightSprite = new Sprite(bitmap)
            this.rightSprite.setFrame(bitmap.width/2, 0, bitmap.width, bitmap.height)
            this.rightSprite.move(bitmap.width/2, 0)
            this.createHorizontalAnimes(bitmap)
            this.addChild(this.leftSprite)
            this.addChild(this.rightSprite)
        }

        createHorizontalAnimes(bitmap){
            let targetValue = 0 - bitmap.width/2
            this.spriteAnimes.push(new Eli.AnimeTiny(this.leftSprite, "x", targetValue, this.parameters.duration, this.parameters.easing, "normal", 0))

            targetValue = Graphics.width
            this.spriteAnimes.push(new Eli.AnimeTiny(this.rightSprite, "x", targetValue, this.parameters.duration, this.parameters.easing, "normal", 0))
        }

        createSplitVertical(bitmap){
            this.topSprite = new Sprite(bitmap)
            this.topSprite.setFrame(0, 0, bitmap.width, bitmap.height/2)
            this.bottomSprite = new Sprite(bitmap)
            this.bottomSprite.setFrame(0, bitmap.height/2, bitmap.width, bitmap.height)
            this.bottomSprite.move(0, bitmap.height/2)
            this.createVerticalAnimes(bitmap)
            this.addChild(this.topSprite)
            this.addChild(this.bottomSprite)
        }

        createVerticalAnimes(bitmap){
            let targetValue = 0 - bitmap.height/2
            this.spriteAnimes.push(new Eli.AnimeTiny(this.topSprite, "y", targetValue, this.parameters.duration, this.parameters.easing, "normal", 0))

            targetValue = Graphics.height
            this.spriteAnimes.push(new Eli.AnimeTiny(this.bottomSprite, "y", targetValue, this.parameters.duration, this.parameters.easing, "normal", 0))
        }

        createSplitHorzAndVert(bitmap){
            const halfWidth = bitmap.width/2
            const halfHeight = bitmap.height/2

            this.topLeftSprite = new Sprite(bitmap)
            this.topLeftSprite.setFrame(0, 0, halfWidth, halfHeight)

            this.topRightSprite = new Sprite(bitmap)
            this.topRightSprite.setFrame(halfWidth, 0, bitmap.width, halfHeight)
            this.topRightSprite.move(halfWidth, 0)

            this.bottomLeftSprite = new Sprite(bitmap)
            this.bottomLeftSprite.setFrame(0, halfHeight, halfWidth, bitmap.height)
            this.bottomLeftSprite.move(0, halfHeight)

            this.bottomRightSprite = new Sprite(bitmap)
            this.bottomRightSprite.setFrame(halfWidth, halfHeight, bitmap.width, bitmap.height)
            this.bottomRightSprite.move(halfWidth, halfHeight)
            this.createHorzAndVertAnimes(bitmap)

            this.addChild(this.topLeftSprite)
            this.addChild(this.topRightSprite)
            this.addChild(this.bottomLeftSprite)
            this.addChild(this.bottomRightSprite)
        }

        createHorzAndVertAnimes(bitmap){
            const {duration, easing} = this.parameters
            let targetX, targetY

            targetX = 0 - bitmap.width/2
            targetY = 0 - bitmap.height/2
            this.spriteAnimes.push(new Eli.AnimeTiny(this.topLeftSprite, "x", targetX, duration, easing, "normal", 0))
            this.spriteAnimes.push(new Eli.AnimeTiny(this.topLeftSprite, "y", targetY, duration, easing, "normal", 0))

            targetX = Graphics.width
            targetY = 0 - bitmap.height/2
            this.spriteAnimes.push(new Eli.AnimeTiny(this.topRightSprite, "x", targetX, duration, easing, "normal", 0))
            this.spriteAnimes.push(new Eli.AnimeTiny(this.topRightSprite, "y", targetY, duration, easing, "normal", 0))

            targetX = 0 - bitmap.width/2
            targetY = Graphics.height
            this.spriteAnimes.push(new Eli.AnimeTiny(this.bottomLeftSprite, "x", targetX, duration, easing, "normal", 0))
            this.spriteAnimes.push(new Eli.AnimeTiny(this.bottomLeftSprite, "y", targetY, duration, easing, "normal", 0))

            targetX = Graphics.width
            targetY = Graphics.height
            this.spriteAnimes.push(new Eli.AnimeTiny(this.bottomRightSprite, "x", targetX, duration, easing, "normal", 0))
            this.spriteAnimes.push(new Eli.AnimeTiny(this.bottomRightSprite, "y", targetY, duration, easing, "normal", 0))
        }

        isRunning(){
            return this.areSpriteAnimesRunning() || this.isFadeSpriteAnimeRunning()
        }

        areSpriteAnimesRunning(){
            return this.spriteAnimes.some(anime => anime.isRunning())
        }

        isFadeSpriteAnimeRunning(){
            return this.fadeSprite.isRunning()
        }

        update(){
            super.update()
            for(const anime of this.spriteAnimes){
                anime.update()
            }
            if(!this.areSpriteAnimesRunning() && this.fadeSprite.alphaAnime.isPaused()){
                this.fadeSprite.alphaAnime.resume()
            }
        }
    },

    Sprite_MoveScreenTransition: class Sprite_MoveScreenTransition extends Sprite{

        initialize(parameters, bitmap){
            super.initialize()
            this.createFadeSprite()
            this.createScreenMoveSprite(bitmap)
            this.createAnime(parameters, bitmap)
        }

        createFadeSprite(){
            this.fadeSprite = new Eli.MapTransitions.Sprite_Fade(Eli.MapTransitions.getParam().sceneDefaultTransition.data, Eli.MapTransitions.fadeInBitmap, false)
            this.fadeSprite.alphaAnime.pause()
            this.addChild(this.fadeSprite)
        }

        createScreenMoveSprite(bitmap){
            this.screenMoveSprite = new Sprite(bitmap)
            this.addChild(this.screenMoveSprite)
        }

        createAnime(parameters, bitmap){
            const [propName, targetValue] = {
                moveScreenLeft: ["x", 0-bitmap.width],
                moveScreenRight: ["x", Graphics.width],
                moveScreenTop: ["y", 0-bitmap.height],
                moveScreenBottom: ["y", Graphics.height],
            }[parameters.type]

            this.positionAnime = new Eli.AnimeTiny(this.screenMoveSprite, propName, targetValue, parameters.data.duration, parameters.data.easing, "normal", 0)
        }

        isRunning(){
            return this.positionAnime.isRunning() || this.isFadeSpriteAnimeRunning()
        }

        isFadeSpriteAnimeRunning(){
            return this.fadeSprite.isRunning()
        }

        update(){
            super.update()
            this.positionAnime.update()
            if(!this.positionAnime.isRunning() && this.fadeSprite.alphaAnime.isPaused()){
                this.fadeSprite.alphaAnime.resume()
            }
        }
    },


    initialize(){
        this.initParameters()
        this.initPluginCommands()
    },

    initParameters(){
        const parameters = PluginManager.parameters(Eli.PluginManager.getPluginName())
        this.parameters = new this.Parameters(parameters)
    },

    initPluginCommands(){
        const commands = []
        Eli.PluginManager.registerCommands(this, commands)
    },

    getParam(){
        return this.parameters
    },
}

{

const Plugin = Eli.MapTransitions
const Alias = {}

Plugin.initialize()

/* -------------------------------- SCENE MAP ------------------------------- */
Scene_Map.prototype.startFadeIn = function(duration, white){}
Scene_Map.prototype.startFadeOut = function(duration, white){}

Scene_Map.prototype.isFading = function() {
    return this.isTransitionRunning()
}

Scene_Map.prototype.isTransitionRunning = function() {
    return this.transContainer.isRunning()
}

Alias.Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects
Scene_Map.prototype.createDisplayObjects = function() {
    Alias.Scene_Map_createDisplayObjects.call(this)
    this.createTransitionContainer()

    if(SceneManager._previousClass.name === "Scene_Map" || !Plugin.fadeOutBitmap){
        this.createTransitionSprites()
    }
}

Scene_Map.prototype.createTransitionContainer = function() {
    Plugin.fadeInBitmap = Bitmap.snap(this)
    this.transContainer = new Plugin.Container()
}

Scene_Map.prototype.createTransitionSprites = function() {
    const transId = ($dataMap.meta.Transition || Plugin.getParam().sceneDefaultTransition.id).toLowerCase().replaceAll(" ", "")
    const transition = this.findTransition(transId)

    if(!Plugin.fadeOutBitmap){
        this.transInSprite = new Plugin.Sprite_Fade(Plugin.getParam().sceneDefaultTransition.data, Plugin.fadeInBitmap, false)
        this.transContainer.setInSprite(this.transInSprite)
        this.transContainer.update()

    }else if(transition.type === "fade"){
        this.createFadeTransition(transition)
        this.transContainer.update()

    }else if(transition.type === "crossFade"){
        this.createCrossFadeTransition(transition)
        this.transContainer.update()

    }else if(transition.type === "zoomCrossFade"){
        this.createZoomCrossFadeTransition(transition)

    }else if(transition.type === "zoom"){
        this.createZoomTransition(transition)

    }else if(transition.type === "pixel"){
        this.createPixelTransition(transition)

    }else if(["splitHorizontal", "splitVertical", "splitHorzAndVert"].includes(transition.type)){
        this.createSplitScreenTransition(transition)

    }else if(["moveScreenLeft", "moveScreenRight", "moveScreenTop", "moveScreenBottom"].includes(transition.type)){
        this.createMoveScreenTransition(transition)
    }
}

Scene_Map.prototype.findTransition = function(transId) {
    const transitionList = Plugin.getParam().transitionList
    const varParam = Plugin.getParam().varInTransition
    let transition = null

    if(varParam > 0){
        const varId = String($gameVariables.value(varParam)).toLowerCase().replaceAll(" ", "")

        if(varId !== "-1"){
            transition = transitionList[varId] || transitionList.find(item => item.id === varId)

        }else{
            transition = transitionList[transId] || transitionList.find(item => item.id === transId)
        }

    }else{
            transition = transitionList[transId] || transitionList.find(item => item.id === transId)
    }

    return transition || Plugin.getParam().sceneDefaultTransition
}

Scene_Map.prototype.createFadeTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_Fade(transition.data, Plugin.fadeOutBitmap, true)
    this.transContainer.setOutSprite(this.transOutSprite)
    
    this.transInSprite = new Plugin.Sprite_Fade(transition.data, Plugin.fadeInBitmap, false)
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createCrossFadeTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_CrossFade(transition.data, Plugin.fadeOutBitmap, true)
    this.transContainer.setOutSprite(this.transOutSprite)
    
    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createZoomCrossFadeTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_ZoomCrossFade(transition.data, Plugin.fadeOutBitmap)
    this.transContainer.setOutSprite(this.transOutSprite)
    
    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createZoomTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_Zoom(transition.data, Plugin.fadeOutBitmap)
    this.transContainer.setOutSprite(this.transOutSprite)
    
    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createPixelTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_Pixel(transition.data, Plugin.fadeOutBitmap)
    this.transContainer.setOutSprite(this.transOutSprite)
    
    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createSplitScreenTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_SplitTransition(transition, Plugin.fadeOutBitmap)
    this.transContainer.setOutSprite(this.transOutSprite)

    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Scene_Map.prototype.createMoveScreenTransition = function(transition) {
    this.transOutSprite = new Plugin.Sprite_MoveScreenTransition(transition, Plugin.fadeOutBitmap)
    this.transContainer.setOutSprite(this.transOutSprite)

    this.transInSprite = new Plugin.Sprite_Dummy()
    this.transContainer.setInSprite(this.transInSprite)
}

Alias.Scene_Map_start = Scene_Map.prototype.start
Scene_Map.prototype.start = function() {
    this.isCustomFading = true
    Alias.Scene_Map_start.call(this)
    this.isCustomFading = false
    this.addChild(this.transContainer)
}

Alias.Scene_Map_onTransferEnd = Scene_Map.prototype.onTransferEnd
Scene_Map.prototype.onTransferEnd = function() {
    if(!this.isCustomFading){
        Alias.Scene_Map_onTransferEnd.call(this)
    }
}

Alias.Scene_Map_terminate = Scene_Map.prototype.terminate
Scene_Map.prototype.terminate = function() {
    Alias.Scene_Map_terminate.call(this)
    this.setSceneTransitionImage()
}

Scene_Map.prototype.setSceneTransitionImage = function() {
    Plugin.fadeOutBitmap = new Bitmap.snap(this)
}

}