//============================================================================
// Eli_BackgroundManagerPro.js
//============================================================================

/*:
@target MZ
@base EliMZ_Book

@plugindesc ♦1.1.2♦ Add customized backgrounds for title, menu scenes, and windows!
@author Hakuen Studio
@url https://hakuenstudio.itch.io/hakuen-studio-background-manager-for-rpg-maker-mv-mz/rate?source=game

@help
★★★★★ Rate the plugin! Please, is very important to me ^^
● Terms of Use
https://www.hakuenstudio.com/terms-of-use-5-0-0
==============================================================================
Plugin Requirements
==============================================================================

Need Eli Book.
Order After Eli Book.

==============================================================================
Features
==============================================================================

● Add customized background to any scene and window!
● Backgrounds can be Spritesheets or parallax type!
● Backgrounds can shown o hide according to conditions!

==============================================================================
Plugin Parameters
==============================================================================

https://docs.google.com/document/d/15BzPclaT3Gtt3IYUjUr_QxIi29q_HbpSxvCh_yohOFs/edit?usp=sharing

==============================================================================

@param sceneBackgrounds
@text Scene Backgrounds
@type struct<sceneBackgroundSt>[]
@desc Here you can build all your background sprites for a specific scene.
@default []

@param titleBackgrounds
@text Title Backgrounds
@type struct<backgroundSt>[]
@desc Here you can build all your background images for the title screen.
@default []

@param hideDefaultTitleBackground
@text Hide Default Title Backgrounds
@type boolean
@desc Set to true if you want to hide the default title backgrounds.
@default true
@parent titleBackgrounds

@param menuBackgrounds
@text Menu Backgrounds
@type struct<backgroundSt>[]
@desc Here you can build all your background sprites for all the menu.
@default []

@param hideDefaultMenuBackground
@text Hide Default Menu Background
@type boolean
@desc Set to true if you want to hide the default menu backgrounds.
@default true
@parent menuBackgrounds

@param winBackgrounds
@text Window Backgrounds
@type struct<winBackgroundSt>[]
@desc Build the window backgrounds here.
@default []

@param hideDarkRectangles
@text Hide Dark Rectangles
@type boolean
@desc Set to true if you want to hide the default dark rectangles on selectable windows when a background exist.
@default true
@parent winBackgrounds

*/

/* ---------------------------- SCENE BACKGROUND ---------------------------- */
{ 
/*~struct~sceneBackgroundSt:

@param sceneList
@text Scene List
@type combo[]
@option Scene_Battle @option Scene_Debug @option Scene_Equip @option Scene_GameEnd @option Scene_Gameover @option Scene_Item @option Scene_Load @option Scene_Map @option Scene_MapSelect @option Scene_Menu @option Scene_MenuInfo @option Scene_Minimap @option Scene_Name @option Scene_Options @option Scene_Save @option Scene_Shop @option Scene_Skill @option Scene_SoundTest @option Scene_Status @option Scene_Title
@desc A list of all scenes that will use this background.
It is case sensitive.
@default []

@param backgroundList
@text Backgrounds
@type struct<backgroundSt>[]
@desc Here you can build all your background sprites for the menu.
@default []

*/
}

/* ----------------------------- WIN BACKGROUND ----------------------------- */
{

/*~struct~winBackgroundSt:

@param windowList
@text Window List
@type combo[]
@option Window_ActorCommand @option Window_BattleActor @option Window_BattleEnemy @option Window_BattleItem @option Window_BattleLog @option Window_BattleSkill @option Window_BattleStatus @option Window_ChoiceList @option Window_CommandInfo @option Window_DebugEdit @option Window_DebugRange @option Window_DescriptionInfo @option Window_EquipCommand @option Window_EquipItem @option Window_EquipSlot @option Window_EquipStatus @option Window_EventItem @option Window_FaceMessage @option Window_GameEnd @option Window_Gold @option Window_Help @option Window_HelpActorCommand @option Window_HelpChoice @option Window_HelpNumberInput @option Window_HelpPartyCommand @option Window_HelpSelectItem @option Window_HelpTitle @option Window_ItemCategory @option Window_ItemList @option Window_LoadPoint @option Window_MapName @option Window_MapSelectCommand @option Window_MenuActor @option Window_MenuCommand @option Window_MenuStatus @option Window_Message @option Window_Minimap @option Window_NameBox @option Window_NameEdit @option Window_NameInput @option Window_NumberInput @option Window_Options @option Window_PartyCommand @option Window_Preview @option Window_SavefileList @option Window_SavePoint @option Window_ScrollText @option Window_ShopBuy @option Window_ShopCommand @option Window_ShopNumber @option Window_ShopSell @option Window_ShopStatus @option Window_SkillList @option Window_SkillStatus @option Window_SkillType @option Window_SoundList @option Window_SoundMainCategory @option Window_SoundPlaying @option Window_SoundSceneTitle @option Window_SoundSubCategory @option Window_Status @option Window_StatusEquip @option Window_StatusParam @option Window_TitleCommand @option Window_TitleInfo @option Window_ToastInfo @option Window_GameFilterHelp @option Window_Options_GameFilter
@desc A list of all windows that will use this background.
It is case sensitive.
@default []

@param backgroundList
@text Backgrounds
@type struct<backgroundSt>[]
@desc Here you can build all your background sprites for the menu.
@default []

*/

}

/* ------------------------------- BACKGROUNDS ------------------------------ */
{

/*~struct~backgroundSt:

@param file
@text Image File
@type file
@dir img/menuBackgrounds
@desc The image used as background.
@default

@param type
@text Type
@type select
@option Sprite
@option Parallax
@desc The background type.
@default Sprite

@param condition
@text Condition
@type multiline_string
@desc A valid javascript formula to be applied as condition to show this background. See help file.
@default return true

@param zIndex
@text Z Index
@type text
@desc The layer index of this background. Does not work for Window Backgrounds.
@default 0

@param --- Sprite ---

@param maxColumns
@text Max Columns
@type number
@min 1
@max 99
@desc The max number of columns/frames this sprite have.
@default 1
@parent --- Sprite ---

@param interval
@text Interval
@type number
@min 1
@desc The time interval required to change the sprite frame.
@default 30
@parent --- Sprite ---

@param position
@text Position
@type struct<positionSt>
@desc The position of the background on the scene.
@default {"alignX":"left","offsetX":"0","alignY":"top","offsetY":"0"}
@parent --- Sprite ---

@param --- Parallax ---

@param scrollX
@text Scroll X
@type number
@desc How much pixels per frame the parallax will move horizontally.
@default 0
@parent --- Parallax ---

@param scrollY
@text Scroll Y
@type number
@desc How much pixels per frame the parallax will move vertically.
@default 0
@parent --- Parallax ---

*/

}

/* -------------------------------- POSITION -------------------------------- */
{
/*~struct~positionSt:

@param alignX
@text Align X
@type select
@option none
@option left
@option center
@option right
@desc Select none to only use offset value.
@default left

@param offsetX
@text Offset X
@type text
@desc The Offset X position.
@default 10
@parent alignX

@param alignY
@text Align Y
@type select
@option none
@option top
@option center
@option bottom
@desc Select none to only use offset value.
@default top

@param offsetY
@text Offset Y
@type text
@desc The offset Y position.
@default 10
@parent alignY

*/
}

"use strict"

var Eli = Eli || {}
var Imported = Imported || {}
Imported.Eli_BackgroundManager = true

/* ========================================================================== */
/*                                   PLUGIN                                   */
/* ========================================================================== */
Eli.BackgroundManager = {

    url: 'https://hakuenstudio.itch.io/hakuen-studio-background-manager-for-rpg-maker-mv-mz',
    pro: true,
    needResetCurrentSettings: false,
    Container_Background: class extends PIXI.Container{

        update(){
            for(const child of this.children){
                child.update()
            }
        }
    },
    Sprite_Background: class extends Sprite {

        initialize(bitmap, parameters){
            super.initialize(bitmap)
            this.initProps(parameters)
            this.updateVisibility()
            this.bitmap.addLoadListener(this.onMainBitmapLoad.bind(this))
        }
    
        initProps(parameters){
            this.parameters = parameters
            this.maxIndex = Math.max(this.parameters.maxColumns - 1, 0)
            this._zIndex = parameters.zIndex
        }
    
        onMainBitmapLoad(){
            this.changeFrame()
            this.setPosition()
        }
    
        changeFrame() {
            const width = this.patternWidth()
            const height = this.bitmap.height
            const x = this.getIndex() * width
            const y = 0
    
            this.setFrame(x, y, width, height)
        }
    
        setPosition(){
            const {alignX, alignY, offsetX, offsetY} = this.parameters.position
            const x = Eli.Utils.calculateScreenPosition(alignX, offsetX, this.bitmap.width/this.parameters.maxColumns, "x")
            const y = Eli.Utils.calculateScreenPosition(alignY, offsetY, this.bitmap.height, "y")
    
            this.move(x, y)
        }
    
        patternWidth() {
            return this.bitmap.width / this.parameters.maxColumns
        }
    
        getIndex(){
            return this.parameters.current.index
        }
    
        update(){
            super.update()
            this.updateVisibility()
    
            if(this.bitmap && this.bitmap.isReady()){
                this.updateIndex()
            }
        }
    
        updateVisibility(){
            this.visible = this.parameters.condition()
        }
    
        updateIndex(){
            this.parameters.current.animationCount++
            
            if(this.canChangeIndex()){
                this.changeIndex()
                this.resetAnimationCount()
                this.changeFrame()
            }
        }
    
        canChangeIndex(){
            return this.parameters.current.animationCount >= this.parameters.interval
        }
    
        changeIndex(){
            if(this.parameters.current.index === this.maxIndex){
                this.parameters.current.index = 0
            }else{
                this.parameters.current.index++
            }
        }
    
        resetAnimationCount(){
            this.parameters.current.animationCount = 0
        }
    },
    Parallax_Background: class extends TilingSprite {

        initialize(bitmap, parameters){
            super.initialize(bitmap)
            this.initProps(parameters)
            this.updateVisibility()
            this.restoreOrigin()
            this.move(0, 0, Graphics.width, Graphics.height)
        }
    
        initProps(parameters){
            this.parameters = parameters
            this.originX = this.origin.x
            this.originY = this.origin.y
            this._zIndex = parameters.zIndex
        }
    
        restoreOrigin(){
            this.origin.x = this.parameters.current.originX
            this.origin.y = this.parameters.current.originY
        }
    
        update(){
            super.update()
            this.updateVisibility()
    
            if(this.bitmap && this.bitmap.isReady()){
                this.changeOrigin()
            }
        }
    
        updateVisibility(){
            this.visible = this.parameters.condition()
        }
    
        changeOrigin(){
            this.origin.x -= this.parameters.scrollX
            this.origin.y -= this.parameters.scrollY
            this.parameters.current.originX -= this.parameters.scrollX
            this.parameters.current.originY -= this.parameters.scrollY
        }
    },
    Parameters: class {
        constructor(parameters){
            this.menu = {
                backgroundList: this.parseBackgroundList(JSON.parse(parameters.menuBackgrounds)),
                hideDefaultBackground: parameters.hideDefaultMenuBackground === "true",
            }
            this.title = {
                backgroundList: this.parseBackgroundList(JSON.parse(parameters.titleBackgrounds)),
                hideDefaultBackground: parameters.hideDefaultTitleBackground === "true",
            }
            this.windows = this.parseCustomBackground(JSON.parse(parameters.winBackgrounds), "windowList")
            this.scenes = this.parseCustomBackground(JSON.parse(parameters.sceneBackgrounds), "sceneList")
            this.hideDarkRectangles = parameters.hideDarkRectangles === "true"
        }

        parseBackgroundList(parameters){
            const sprites = []
            const parallax = []
    
            for(const param of parameters){
                const background = this.parseBackground(JSON.parse(param))
    
                if(background.type === "Sprite"){
                    sprites.push(background)
                }else{
                    parallax.push(background)
                }
            }
    
            return [...parallax, ...sprites]
        }
    
        parseBackground(background){
            const position = JSON.parse(background.position)

            return {
                file: background.file,
                maxColumns: Number(background.maxColumns),
                interval: Number(background.interval),
                scrollX: Number(background.scrollX),
                scrollY: Number(background.scrollY),
                type: background.type,
                condition: new Function(background.condition || `return true`),
                zIndex: Number(background.zIndex || "0"),
                position: {
                    alignX: position.alignX, 
                    offsetX: Number(position.offsetX),
                    alignY: position.alignY,
                    offsetY: Number(position.offsetY),
                },
                current: {
                    originX: 0,
                    originY: 0,
                    index: 0,
                    animationCount: 0,
                }
            }
        }
    
        parseCustomBackground(parameters, listKey){
            const customBackgrounds = []
    
            for(const param of parameters){
                const data = JSON.parse(param)
                const obj = {
                    nameList: JSON.parse(data[listKey]),
                    backgroundList: this.parseBackgroundList(JSON.parse(data.backgroundList))
                }
    
                customBackgrounds.push(obj)
            }
    
            return customBackgrounds
        }
    },

    initialize(){
        this.initParameters()
    },

    initParameters(){
        const parameters = PluginManager.parameters(Eli.PluginManager.getPluginName())
        this.parameters = new this.Parameters(parameters)
    },

    getParam(){
        return this.parameters
    },

    resetCurrentMenuSettings(){
        for(const param of this.getParam().menu.backgroundList){
            param.current.originX = 0
            param.current.originY = 0
            param.current.index = 0
            param.current.animationCount = 0
        }
    },
}

{

const Plugin = Eli.BackgroundManager
const Alias = {}
const FOLDER = "img/menuBackgrounds/"

Plugin.initialize()

/* ------------------------------- SCENE BASE ------------------------------- */
Alias.Scene_Base_initialize = Scene_Base.prototype.initialize
Scene_Base.prototype.initialize = function(){
    Alias.Scene_Base_initialize.call(this)
    this.dynamicBackgrounds = []
}

Alias.Scene_Base_create = Scene_Base.prototype.create
Scene_Base.prototype.create = function() {
    Alias.Scene_Base_create.call(this)
    this.createBackgroundContainer()
    this.createDynamicBackgrounds()
}

Scene_Base.prototype.createBackgroundContainer = function() {
    this.backgroundContainer = new Plugin.Container_Background()
    this.addChild(this.backgroundContainer)
}

Scene_Base.prototype.canCreateDynamicBackgrounds = function() {
    return !(this instanceof Scene_Battle || this instanceof Scene_Map)
}

Scene_Base.prototype.createDynamicBackgrounds = function() {
    if(this.canCreateDynamicBackgrounds()){
        this.createSceneDynamicBackgrounds()
    }
    this.backgroundContainer.sortChildren()
}

Scene_Base.prototype.createSceneDynamicBackgrounds = function() {
    const emptyList = {backgroundList: []}
    const mainParameter = this.findSceneBackground() || emptyList

    for(let i = 0; i < mainParameter.backgroundList.length; i++){
        const parameters = mainParameter.backgroundList[i]
        const bitmap = ImageManager.loadBitmap(FOLDER, parameters.file)

        parameters.current.originX = 0
        parameters.current.originY = 0
        parameters.current.index = 0
        parameters.current.animationCount = 0

        if(parameters.type === "Sprite"){
            var sprite = new Plugin.Sprite_Background(bitmap, parameters)
        }else{
            var sprite = new Plugin.Parallax_Background(bitmap, parameters)
            sprite.move(0, 0, Graphics.width, Graphics.height)
        }

        this.dynamicBackgrounds.push(sprite)
        this.backgroundContainer.addChild(sprite)
    }
}

Scene_Base.prototype.findSceneBackground = function() {
    const name = this.constructor.name
    return Plugin.getParam().scenes.find(item => item.nameList.includes(name))
}

Scene_Base.prototype.getDynamicBackgrounds = function() {
    return this.dynamicBackgrounds
}

/* ------------------------------- SCENE TITLE ------------------------------ */
Alias.Scene_Title_createBackground = Scene_Title.prototype.createBackground
Scene_Title.prototype.createBackground = function() {
    Alias.Scene_Title_createBackground.call(this)

    if(Plugin.getParam().title.hideDefaultBackground){
        this._backSprite1.visible = false
        this._backSprite2.visible = false
    }

    this.setChildIndex(this._backSprite1, 0)
    this.setChildIndex(this._backSprite2, 1)
}

Alias.Scene_Title_createDynamicBackgrounds = Scene_Title.prototype.createDynamicBackgrounds
Scene_Title.prototype.createDynamicBackgrounds = function() {
    this.createDynamicTitleBackgrounds()
    Alias.Scene_Title_createDynamicBackgrounds.call(this)
}

Scene_Title.prototype.createDynamicTitleBackgrounds = function() {
    for(const parameters of Plugin.getParam().title.backgroundList){
        const bitmap = ImageManager.loadBitmap(FOLDER, parameters.file)

        if(parameters.type === "Sprite"){
            var sprite = new Plugin.Sprite_Background(bitmap, parameters)
        }else{
            var sprite = new Plugin.Parallax_Background(bitmap, parameters)
        }
        
        this.dynamicBackgrounds.push(sprite)
        this.backgroundContainer.addChild(sprite)
    }
}

/* -------------------------------- SCENE MAP ------------------------------- */
Alias.Scene_Map_start = Scene_Map.prototype.start
Scene_Map.prototype.start = function() {
    Alias.Scene_Map_start.call(this)
    Plugin.needResetCurrentSettings = true
}

/* ----------------------------- SCENE MENU BASE ---------------------------- */
Alias.Scene_MenuBase_createBackground = Scene_MenuBase.prototype.createBackground
Scene_MenuBase.prototype.createBackground = function() {
    Alias.Scene_MenuBase_createBackground.call(this)

    this.setChildIndex(this._backgroundSprite, 0)

    if(Plugin.getParam().menu.hideDefaultBackground){
        this._backgroundSprite.visible = false
    }
}

Alias.Scene_MenuBase_createDynamicBackgrounds = Scene_MenuBase.prototype.createDynamicBackgrounds
Scene_MenuBase.prototype.createDynamicBackgrounds = function() {
    this.createMenuDynamicBackgrounds()
    Alias.Scene_MenuBase_createDynamicBackgrounds.call(this)
}

Scene_MenuBase.prototype.createMenuDynamicBackgrounds = function() {
    const backgroundParameters = this.findMenuBackgroundParameter()

    for(const parameters of backgroundParameters){
        const bitmap = ImageManager.loadBitmap(FOLDER, parameters.file)

        if(parameters.type === "Sprite"){
            var sprite = new Plugin.Sprite_Background(bitmap, parameters)
        }else{
            var sprite = new Plugin.Parallax_Background(bitmap, parameters)
        }
        
        this.dynamicBackgrounds.push(sprite)
        this.backgroundContainer.addChild(sprite)
    }
}

Scene_MenuBase.prototype.findMenuBackgroundParameter = function(){
    const stack = SceneManager._stack[0]

    if(stack && stack.name && stack.name === "Scene_Title"){
        return Plugin.getParam().title.backgroundList
    }else{
        return Plugin.getParam().menu.backgroundList
    }
}

/* ------------------------------- SCENE MENU ------------------------------- */
Alias.Scene_Menu_create = Scene_Menu.prototype.create
Scene_Menu.prototype.create = function(){
    if(Plugin.needResetCurrentSettings){
        Plugin.needResetCurrentSettings = false
        Plugin.resetCurrentMenuSettings()
    }

    Alias.Scene_Menu_create.call(this)
}

/* ------------------------------- WINDOW BASE ------------------------------ */
Alias.Window_Base_initialize = Window_Base.prototype.initialize
Window_Base.prototype.initialize = function(rect) {
    Alias.Window_Base_initialize.call(this, rect)
    this.createDynamicBackgrounds()
}

Window_Base.prototype.createDynamicBackgrounds = function() {
    const emptyList = {backgroundList: []}
    const mainParameter = this.findWinBackground() || emptyList
    this.dynamicBackgrounds = []

    for(let i = 0; i < mainParameter.backgroundList.length; i++){
        const parameters = mainParameter.backgroundList[i]
        const bitmap = ImageManager.loadBitmap(FOLDER, parameters.file)

        if(parameters.type === "Sprite"){
            var sprite = new Plugin.Sprite_Background(bitmap, parameters)
        }else{
            var sprite = new Plugin.Parallax_Background(bitmap, parameters)
            sprite.move(0, 0, this.innerWidth, this.innerHeight)
        }

        this.dynamicBackgrounds.push(sprite)
        this._clientArea.addChildAt(sprite, i)
    }
}

Window_Base.prototype.findWinBackground = function() {
    const name = this.constructor.name
    return Plugin.getParam().windows.find(item => item.nameList.includes(name))
}

Window_Base.prototype.setBackgroundsToBack = function() {
    for(let i = 0; i < this.dynamicBackgrounds.length; i++){
        const sprite = this.dynamicBackgrounds[i]
        this._clientArea.setChildIndex(sprite, i)
    }
}

/* ---------------------------- WINDOW SELECTABLE --------------------------- */
Alias.Window_Selectable_drawBackgroundRect = Window_Selectable.prototype.drawBackgroundRect
Window_Selectable.prototype.drawBackgroundRect = function(rect) {
    if(!this.canHideDarkRectanglesForBackground()){
        Alias.Window_Selectable_drawBackgroundRect.call(this, rect)
    } 
}

Window_Selectable.prototype.canHideDarkRectanglesForBackground = function() {
    return Plugin.getParam().hideDarkRectangles && this.dynamicBackgrounds.length > 0
}

}