var Imported = Imported || {};
Imported.MultiLevelMap = 1;

//=============================================================================
 /*:
 * @plugindesc v1.0.1
 * Allows the visual stacking of several maps to simulate altitude.
 * <MultiLevelMap>
 *
 * @author Fhntop       Site: https://www.youtube.com/user/fhntop
 *
 * @param Hole Region
 * @desc The region ID describing a hole.
 * Script Default: 101
 * @default 101
 *
 * @param Up Region
 * @desc The region ID describing an upwards movement (Increases Z on touch).
 * Script Default: 102
 * @default 102
 *
 * @param Down Region
 * @desc The region ID describing an downwards movement (Decreases Z on touch).
 * Script Default: 103
 * @default 103
 *
 * @param =====================
 * @desc Spacer
 * @default
 *
 * @param Z Acceleration
 * @desc "Gravity" constant. When falling, it's used to increase the speed of Z each frame
 * Script Default: 0.981
 * @default 0.981
 *
 * @help
 * ============================================================================
 *  Multi Level Map
 *  Author: Fhntop
 *  Version: 1.0.1 - 2017-05-11
 * ============================================================================
 *  [Description]
 *    Allows the visual stacking of several maps to simulate altitude.
 * 
 * 
 * ============================================================================
 *  [How to use]
 *    On the Ground Level Map (Z=0), use <ABOVE L />, with L a list of numbers, 
 *  like:
 *      <ABOVE m1 />
 *      <ABOVE m1, m2, />
 *      <ABOVE m1, m2, m3, m4 />
 *     where m1, m2, m3 and m4 are the Ids of the Maps for Z=1 to Z=4
 *    [UNTESTED] Idem for the lower levels:
 *      <BELOW m1 /> 
 *      <BELOW m1, m2, m3, m4 /> 
 *     where m1, m2, m3 and m4 are the Ids of the Maps for Z=-1 to Z=-4
 *    On those maps, add this on your note:
 *      <BASE_MAP m />
 *     where m is the Id of the Ground Map.
 *    This allows you to place or transport your character to any map,
 *  even if it's not the ground level.
 * 
 *    You can set Regions in each map to "teleport" to the above or below 
 *  Z level, and you can also make holes where the characters will fall
 *  until they reach a "non-hole" spot.
 * 
 *    If you can't move on a specific level, you could try loading that map 
 *  alone to check if you can move there.
 * 
 * 	 Extra Note: 
 *  Each character has a "character._gravityInfluence" that by default it's 1.
 *  If you change it to -1, you can have inverse gravity, though I haven't 
 *  tried it too much. It can also be 0.5 for half the effect, and so on...
 *  
 * 
 * ============================================================================
 *  [How it works]
 *    Expands the x-y coordinates to up to a "virtual" of:
 *      maxX = (2*maxZ + 1) * width
 *      maxY = if minZ < 0 then (3 * height) else (height)
 *    Translates every (x, y) with x,y >= 0 to (x', y', z') such that:
 *      0 <= x' < width
 *      0 <= y' < height
 *    Uses the (x', y') of the stacked map in Z = z'
 *    The map's width and height remain unchanged. 
 *    There is $gameMap.realWidth() and $gameMap.realHeight() to provide the
 *  true limits of the map.
 * 
 *    The events on each level may have different internal ids than those seen
 *  on the Editor. For example:
 *      Z=0 events have Offset(Z=0) = 0, 
 *      Z=1 events have Offset(Z=1) #(Events_In_Z = 0)
 *      Z=2 have Offset(Z=2) = Offset(Z=1) + #(Events_In_Z = 1)
 *  and so on... 
 *    I haven't played too much with the eventing but the events on each Map
 *  should work as if that's the only Map. If you want to change the properties
 *  of another map, or another Map's Event, you would need to do this via 
 *  scripting. The Player is the exception to this, since it has a global id.
 * 
 *    Each level is drawn on their own Tilemap, in ascending order of Z. Each
 *  character on a higher tilemap has a Y offset on the screen depending on the 
 *  Z value. On Z=0, it's 0. On Z > 0, it's (Z * $gameMap.tileHeight())
 *    Each tilemap has its Y origin 1 tile above per Z level.
 *    This is what gives the illusion of height.
 * 
 * 
 * ============================================================================
 *  [Known Issues and Unknown Test Results]
 *    I] You can still move when falling, but hey, maybe that's not an issue?
 *    ?] Having a map with Z lower than 0 hasn't been tested
 *    ?] The destination sprite, the click on screen-pathfinding may not work 
 *  on levels above Z=0
 *    ?] Maybe avoid using <ABOVE x /> on a Map with Id=x.
 * 	  ?] I've only used sideways staircases. I have no idea what it'll do with 
 *  vertical stairs.
 *    I] Sometimes, going higher than the maxZ can crash the rendering of the 
 *  game (it freezes).
 *    I] Objects don't stack, so you might get stuck if you fall on top of
 *  something.
 * 
 * 
 * ============================================================================
 *  [Terms of Use]
 *    You can use, copy or modify any part of the plugin, provided you mention
 *  the original Plugin (Multi Map Level) and Author (Fhntop).
 *    Any modified version of this Plugin must conform to these Terms of Use 
 *  and should not be commercialized.
 *    You can use this copy, or a modified version in both commercial and 
 *  non-commercial games, as long as you credit the original Author (Fhntop). 
 *    In the case of using a modified version, the Authors of that version can 
 *  also request credit and please also respect their Terms of Use.
 * 
 * 
 * ============================================================================
 *  [Final notes]
 *    Please experiment with it as much as you want! You can modify it as much 
 *  as you'd like. Keep in mind I'm sharing it for everyone to use, so if you 
 *  have a great idea for an improved version and you manage to make it real,
 *  I'd love it if you'd also share it with the community!
 * 
 */
//=============================================================================

var REGION_HOLE = 101;
var REGION_UP = 102;
var REGION_DOWN = 103;
var Z_DESC_ACCEL = 0.981;
var Z_DESC_START_DIST = 0.001;


(function() {
	function processParameters() {
		var parameters   = $plugins.filter(function(p) { return p.description.contains('<MultiLevelMap>') && p.status; })[0].parameters;

    	REGION_HOLE      = Number(parameters['Hole Region'] || 101);
    	REGION_UP        = Number(parameters['Up Region'] || 102);
    	REGION_DOWN      = Number(parameters['Down Region'] || 103);
    	Z_DESC_ACCEL     = Number(parameters['Z Acceleration'] || 0.981);
	}
	processParameters();

	// **************************************************************************
	//
	//                               DataManager
	//
	// **************************************************************************

	// New function
	DataManager.loadDataFileCallback = function(src, okfn, errfn) {
	    var xhr = new XMLHttpRequest();
	    var url = 'data/' + src;
	    xhr.open('GET', url);
	    xhr.overrideMimeType('application/json');
	    xhr.onload = function() {
	        if (xhr.status < 400) {
	        	var data = JSON.parse(xhr.responseText)
	            DataManager.onLoad(data);
	        	okfn.call(this, data);
	        } else {
	        	errfn.call(this, xhr);
	        }
	    };
	    xhr.onerror = function() {
	        errfn.call(this, xhr);
	    };
	    xhr.send();
	}

	// Replaced function
	DataManager.loadMapData = function(mapId) {
		return DataManager.loadMapDataWithCallback(mapId);
	}

	// New function
	DataManager.loadMapDataWithCallback = function(mapId, onLoadFn) {
	    if (mapId > 0) {
	    	window['$dataMap'] = null;

	        var filename = 'Map%1.json'.format(mapId.padZero(3));
	        DataManager.loadDataFileCallback(filename, function(baseMapData) {
	        	var match = baseMapData.note.match(/\<BASE_MAP \s*([0-9]+)\s*\>/i)
	        	var baseMapId = match ? (JSON.parse('' + match[1] + '') || mapId) : mapId;
	        	if (baseMapId != mapId) {
	        		//console.log('LOADING BASE_MAP', baseMapId);
	        		DataManager.loadMapDataWithCallback(baseMapId, function() {
	        			$gamePlayer.setHomeZ(MultiLevelMapManager.getZByMapId(mapId));
	        			$gamePlayer.setZ(MultiLevelMapManager.getZByMapId(mapId));
	        			if (onLoadFn) onLoadFn.call(this, true);
	        		});
	        	} else {
	        		//console.log('LOADING MAP', mapId);
	        		MultiLevelMapManager.setup(baseMapId, baseMapData);

		        	var match = baseMapData.note.match(/\<ABOVE ([^\>]*)\>/i)
		        	var mapsAbove = match ? (JSON.parse('[' + match[1] + ']') || []) : [];

		        	var match = baseMapData.note.match(/\<BELOW ([^\>]*)\>/i)
		        	var mapsBelow = match ? (JSON.parse('[' + match[1] + ']') || []) : [];

		        	var mapsRequired = mapsAbove.length + mapsBelow.length + 1;
		        	var mapLoaded = function() {
		        		mapsRequired --;
		        		if (mapsRequired <= 0) {
		        			if ($gameMap.mapId() != mapId) {
			        			$gamePlayer.setHomeZ(0);
			        			$gamePlayer.setZ(0);
			        		}
		        			window['$dataMap'] = baseMapData;

		        			//console.log('LOADED');
		        			if (onLoadFn) onLoadFn.call(this, true);
		        		}
		        	}


		        	mapsAbove.forEach(function(mapId, i) {
		        		var filename = 'Map%1.json'.format(mapId.padZero(3));
		        		DataManager.loadDataFileCallback(filename, function(data) {
		        			//console.log('LOADED ABOVE', i);
							MultiLevelMapManager.setZMap(i + 1, mapId, data);
							mapLoaded();
		        		}, function (err) {
		        			DataManager._errorUrl = DataManager._errorUrl || 'data/' + filename;
		        			console.log("ERROR loading map above, ID", mapId);

		        			mapLoaded();
		        		})
		        	})

		        	mapsBelow.forEach(function(mapId, i) {
		        		var filename = 'Map%1.json'.format(mapId.padZero(3));
		        		DataManager.loadDataFileCallback(filename, function(data) {
		        			//console.log('LOADED BELOW', i);
							MultiLevelMapManager.setZMap(-(i + 1), mapId, data);
							mapLoaded();
		        		}, function (err) {
		        			DataManager._errorUrl = DataManager._errorUrl || 'data/' + filename;
		        			console.log("ERROR loading map below, ID", mapId);

		        			mapLoaded();
		        		})
		        	})

		        	mapLoaded();
		        }
	        }, function(xhr) {
				DataManager._errorUrl = DataManager._errorUrl || 'data/' + filename;
				if (onLoadFn) onLoadFn.call(this, false);
	        })
	    } else {
	        this.makeEmptyMap();
	        if (onLoadFn) onLoadFn.call(this, false);
	    }
	}



	// **************************************************************************
	//	NEW
	//                           MultiLevelMapManager
	//
	// **************************************************************************

	MultiLevelMapManager = function() { throw Exception('Bad Timmy!')}
	MultiLevelMapManager._zToEventOffset = {};
	MultiLevelMapManager.setup = function(mapId, data) {
		this._mapSetupDone = false;
		this._maxX = data.width;
		this._maxY = data.height;
		this._dataMap = {};
		this._mapIdToZ = {};
		this._zToMapId = {};

		this._dataMap[0] = data
		this._zToMapId[0] = mapId;
		this._mapIdToZ[mapId] = 0;
		this._minZ = 0;
		this._maxZ = 0;
	}
	MultiLevelMapManager.setZMap = function(z, mapId, data) {
		if (z == 0) return;
		if (z > 0) {
			this._maxX = Math.max(this._maxX, this._dataMap[0].width * (1 + z * 2));
		} else if (z < 0) {
			this._maxX = Math.max(this._maxX, this._dataMap[0].width * (1 - z * 2));
			this._maxY = this._dataMap[0].height * 3;
		}
		this._maxZ = Math.max(this._maxZ, z);
		this._minZ = Math.min(this._minZ, z);

		this._dataMap[z] = data;
		this._zToMapId[z] = mapId;
		this._mapIdToZ[mapId] = z;
	}

	MultiLevelMapManager.minZ = function() {
		return this._minZ;
	}
	MultiLevelMapManager.maxZ = function() {
		return this._maxZ;
	}
	MultiLevelMapManager.maxX = function() {
		return this._maxX;
	}
	MultiLevelMapManager.maxY = function() {
		return this._maxY;
	}


	// This prevents the gameMap to think it's loaded when it's not
	MultiLevelMapManager.mapSetupDone = function() {
		this._mapSetupDone = true;
	}
	MultiLevelMapManager.isMapSetupDone = function() {
		return this._mapSetupDone;
	}


	MultiLevelMapManager.getDataMapAtLevel = function(z) {
		if (!z || z == 0) return $dataMap;
		return this._dataMap[z] || null;
	}
	MultiLevelMapManager.hasMapId = function(mapId) {
		if (!this._mapSetupDone) return false;
		if (!this._mapIdToZ) return $gameMap.mapId() == mapId;
		return mapId in this._mapIdToZ;
	}
	MultiLevelMapManager.getMapIdByZ = function(z) {
		if (!this._zToMapId) return 0;
		return (z in this._zToMapId) ? this._zToMapId[z] : 0;
	}
	MultiLevelMapManager.getZByMapId = function(mapId) {
		if (!this._mapIdToZ) return 0;
		return (mapId in this._mapIdToZ) ? this._mapIdToZ[mapId] : 999;
	}
	MultiLevelMapManager.getDataMaps = function() {
		return this._dataMap;
	}

	// Keeps track of the eventId offsets for each Map Level
	MultiLevelMapManager.clearEventIdOffset = function() {
		this._zToEventOffset = {};
	}
	MultiLevelMapManager.setEventIdOffset = function(z, offset) {
		this._zToEventOffset[z] = offset;
	}
	MultiLevelMapManager.eventIdOffset = function(z) {
		return this._zToEventOffset[Math.floor(z || 0)] || 0;
	}
	MultiLevelMapManager.getZByEventId = function(eventId) {
		var ret = 0;
		var self = this;
		MultiLevelMapManager.forSomeLayers(function (z) {
			if (self._zToEventOffset[z] > eventId) return true;
			ret = z;
		});
		return ret;
	}


	MultiLevelMapManager.isValid = function(x, y) {
		if (x < 0) return false;
		if (y < 0) return false;
		if (x >= this._maxX) return false;
		if (y >= this._maxY) return false;

		if (parseInt(x / this._dataMap[0].width) % 2 == 1) return false;
		if (parseInt(y / this._dataMap[0].height) % 2 == 1) return false;
		return true;
	}
	MultiLevelMapManager.isValidZ = function(z) {
		if (!this._zToMapId) return !z || z == 0;
		return Math.floor(z) in this._zToMapId;
	}

	MultiLevelMapManager.xOffset = function(z) {
		if (!z) return 0;
		if (z > 0) return this._dataMap[0].width * (Math.floor(z) * 2)
		return this._dataMap[0].width * (Math.floor(-z) * 2)
	}
	MultiLevelMapManager.yOffset = function(z) {
		if (!z) return 0;
		if (z > 0) return 0;
		return this._dataMap[0].height * 2;
	}
	MultiLevelMapManager.getZFromXY = function(x, y) {
		var z = (y > this._dataMap[0].height) ? -1 : 1;
		z *= parseInt(x / (this._dataMap[0].width * 2))
		return z;
	}
	MultiLevelMapManager.getXFromXY = function(x, y) {
		return x % (this._dataMap[0].width * 2);
	}
	MultiLevelMapManager.getYFromXY = function(x, y) {
		return y % (this._dataMap[0].height * 2);
	}

	MultiLevelMapManager.getEvents = function() {
		var ret = [];
		var self = this;
		MultiLevelMapManager.forAllLayers(function (z) {
			var dataMap = this._dataMap[z];
			ret.push({'events': dataMap.events, 'mapId': self._zToMapId[z], 'z': z});
		});
		return ret;
	}


	MultiLevelMapManager.forAllLayers = function(fn, obj) {
		for (var z = this._minZ; z <= this._maxZ; z++) {
			fn.call(obj || this, z);
		}
	}
	MultiLevelMapManager.forSomeLayers = function(fn, obj) {
		for (var z = this._minZ; z <= this._maxZ; z++) {
			if (fn.call(obj || this, z)) break;
		}
	}







	// **************************************************************************
	//
	//                               Game_Map
	//
	// **************************************************************************

	// Reused function
	_old_Game_Map_prototype_initialize = Game_Map.prototype.initialize;
	Game_Map.prototype.initialize = function() {
		this._layerData = {};
		_old_Game_Map_prototype_initialize.call(this);
	}

	// New function
	Game_Map.prototype.realWidth = function() {
		return MultiLevelMapManager.maxX();
	}
	// New function
	Game_Map.prototype.realHeight = function() {
		return MultiLevelMapManager.maxY();
	}


	// Reused function
	old_Game_Map_prototype_setup = Game_Map.prototype.setup;
	Game_Map.prototype.setup = function(mapId) {
		var allDataMap = MultiLevelMapManager.getDataMaps();
		this._layerData = {};

		for (var z in allDataMap) {
			if (allDataMap.hasOwnProperty(z)) {
				if (z == 0) continue;

				// Copy the data here. The events can change these things ingame
				dataMap = allDataMap[z];
				var layerData = {
					displayName: dataMap.displayName,
					disableDashing: dataMap.disableDashing,
					encounterList: dataMap.encounterList,
					encounterStep: dataMap.encounterStep,
					autoplayBgm: dataMap.autoplayBgm,
					autoplayBgs: dataMap.autoplayBgs,

					tilesetId: dataMap.tilesetId,
					parallaxName: dataMap.parallaxName || '',
				    parallaxLoopX: dataMap.parallaxLoopX,
				    parallaxLoopY: dataMap.parallaxLoopY,
				    parallaxSx: dataMap.parallaxSx,
				    parallaxSy: dataMap.parallaxSy,
					parallaxX: 0,
					parallaxY: 0
				}
				layerData.parallaxZero = ImageManager.isZeroParallax(layerData.parallaxName);

				if (dataMap.specifyBattleback) {
			        layerData.battleback1Name = dataMap.battleback1Name;
			        layerData.battleback2Name = dataMap.battleback2Name;
			    } else {
			        layerData.battleback1Name = null;
			        layerData.battleback2Name = null;
			    }

			    this._layerData[z] = layerData;
			}
		}
		MultiLevelMapManager.mapSetupDone();

		old_Game_Map_prototype_setup.apply(this, arguments);
	}

	// Replaced function
	Game_Map.prototype.tilesetId = function(z) {
		if (z && Math.floor(z) != 0) return this._layerData[Math.floor(z)].tilesetId;
	    return this._tilesetId;
	}

	// Replaced function
	Game_Map.prototype.parallaxName = function(z) {
		if (z && Math.floor(z) != 0) return this._layerData[Math.floor(z)].parallaxName;
	    return this._parallaxName;
	}

	// Replaced function
	Game_Map.prototype.battleback1Name = function() {
		if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].battleback1Name;
	    return this._battleback1Name;
	}

	// Replaced function
	Game_Map.prototype.battleback2Name = function() {
		if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].battleback2Name;
	    return this._battleback2Name;
	}

	// Replaced function
	Game_Map.prototype.isNameDisplayEnabled = function() {
		if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].nameDisplay;
	    return this._nameDisplay;
	}

	// Replaced function
	Game_Map.prototype.disableNameDisplay = function(z) {
		if (z && Math.floor(z) != 0) {	this._layerData[Math.floor(z)].nameDisplay = false;	}
		else   			 {	this._nameDisplay = false;	}
	}

	// Replaced function
	Game_Map.prototype.enableNameDisplay = function(z) {
	    if (z && Math.floor(z) != 0) {	this._layerData[Math.floor(z)].nameDisplay = true;	}
		else   			 {	this._nameDisplay = true;	}
	}

	// Replaced function
	Game_Map.prototype.setupEvents = function() {
		this._events = [];
		
		var i = 0;
		var self = this;
		var mapEvents = MultiLevelMapManager.getEvents();
		MultiLevelMapManager.clearEventIdOffset();
		mapEvents.forEach(function(mapEventDesc) {
			var events = mapEventDesc.events;
			var mapId = mapEventDesc.mapId;
			var z = mapEventDesc.z;

			MultiLevelMapManager.setEventIdOffset(z, i);
			events.forEach(function(event) {
				if (event)	self._events[i] = new Game_Event(mapId, i);
				i ++;
			});
		})

	    this._commonEvents = this.parallelCommonEvents().map(function(commonEvent) {
	        return new Game_CommonEvent(commonEvent.id);
	    });
	    this.refreshTileEvents();
	}

	// Replaced function
	Game_Map.prototype.event = function(eventId, z) {
	    return this._events[MultiLevelMapManager.eventIdOffset(Math.floor(z)) + eventId];
	}

	// Replaced function
	Game_Map.prototype.eraseEvent = function(eventId, z) {
	    this._events[MultiLevelMapManager.eventIdOffset(Math.floor(z)) + eventId].erase();
	}

	// Replaced function
	Game_Map.prototype.setDisplayPos = function(x, y) {
	    if (this.isLoopHorizontal()) {
	        this._displayX = x.mod(this.width());
	        this._parallaxX = x;
	    } else {
	        var endX = this.width() - this.screenTileX();
	        this._displayX = endX < 0 ? endX / 2 : x.clamp(0, endX);
	        this._parallaxX = this._displayX;
	    }
	    if (this.isLoopVertical()) {
	        this._displayY = y.mod(this.height());
	        this._parallaxY = y;
	    } else {
	        var endY = this.height() - this.screenTileY();
	        this._displayY = endY < 0 ? endY / 2 : y.clamp(0, endY);
	        this._parallaxY = this._displayY;
	    }
	}

	// Replaced function
	Game_Map.prototype.tileset = function(z) {
		return $dataTilesets[this.tilesetId(z)];
	}

	// Replaced function
	Game_Map.prototype.tilesetFlags = function(z) {
	    var tileset = this.tileset(z);
	    if (tileset) {
	        return tileset.flags;
	    } else {
	        return [];
	    }
	}

	// Replaced function
	Game_Map.prototype.displayName = function() {
		if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].displayName;
	    return $dataMap.displayName;
	}

	// Replaced function
	Game_Map.prototype.isDashDisabled = function() {
	    if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].disableDashing;
	    return $dataMap.disableDashing;
	}

	// Replaced function
	Game_Map.prototype.encounterList = function() {
	    if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].encounterList;
	    return $dataMap.encounterList;
	}

	// Replaced function
	Game_Map.prototype.encounterStep = function() {
	    if ($gamePlayer.z && Math.floor($gamePlayer.z) != 0) return this._layerData[Math.floor($gamePlayer.z)].encounterStep;
	    return $dataMap.encounterStep;
	}

	// Replaced function
	Game_Map.prototype.isOverworld = function() {
	    return this.tileset($gamePlayer.z) && this.tileset($gamePlayer.z).mode === 0;
	}

	// Reused function
	old_Game_Map_prototype_adjustX = Game_Map.prototype.adjustX;
	Game_Map.prototype.adjustX = function(x) {
		return old_Game_Map_prototype_adjustX.call(this, MultiLevelMapManager.getXFromXY(x, 0));
	}

	// Reused function
	old_Game_Map_prototype_adjustY = Game_Map.prototype.adjustY;
	Game_Map.prototype.adjustY = function(y) {
		return old_Game_Map_prototype_adjustY.call(this, MultiLevelMapManager.getYFromXY(0, y));
	}

	// Replaced function
	Game_Map.prototype.canvasToMapY = function(y, z) {
		if (arguments.length <= 1) z = $gamePlayer.z;

	    var tileHeight = this.tileHeight();
	    var originY = (z + this._displayY) * tileHeight;
	    var mapY = Math.floor((originY + y) / tileHeight);
	    return this.roundY(mapY);
	}

	// I kept this here, if you rename it "autoplay" it should play the bgm/bgs of the current 
	// map instead of the Z=0 map
	Game_Map.prototype._______________autoplay = function() {
		var data = $gamePlayer.z ? this._layerData[$gamePlayer.z] : $dataMap;
	    if (data.autoplayBgm) {
	        AudioManager.playBgm(data.bgm);
	    }
	    if (data.autoplayBgs) {
	        AudioManager.playBgs(data.bgs);
	    }
	}

	// Reused function
	old_Game_Map_prototype_eventsXy = Game_Map.prototype.eventsXy;
	Game_Map.prototype.eventsXy = function(x, y, z) {
		var ox = MultiLevelMapManager.xOffset(z);
		var oy = MultiLevelMapManager.yOffset(z);
		return old_Game_Map_prototype_eventsXy.call(this, ox + x, oy + y);
	}

	// Reused function
	old_Game_Map_prototype_eventsXyNt = Game_Map.prototype.eventsXyNt;
	Game_Map.prototype.eventsXyNt = function(x, y, z) {
		var ox = MultiLevelMapManager.xOffset(z);
		var oy = MultiLevelMapManager.yOffset(z);
		return old_Game_Map_prototype_eventsXyNt.call(this, ox + x, oy + y);
	}

	// Reused function
	old_Game_Map_prototype_tileEventsXy = Game_Map.prototype.tileEventsXy;
	Game_Map.prototype.tileEventsXy = function(x, y, z) {
		var ox = MultiLevelMapManager.xOffset(z);
		var oy = MultiLevelMapManager.yOffset(z);
		return old_Game_Map_prototype_tileEventsXy.call(this, ox + x, oy + y);
	}

	// Reused function
	old_Game_Map_prototype_eventIdXy = Game_Map.prototype.eventIdXy;
	Game_Map.prototype.eventIdXy = function(x, y, z) {
		var ox = MultiLevelMapManager.xOffset(z);
		var oy = MultiLevelMapManager.yOffset(z);
		return old_Game_Map_prototype_eventIdXy.call(this, ox + x, oy + y);
	}

	// Replaced function
	Game_Map.prototype.isValid = function(x, y) {
		return MultiLevelMapManager.isValid(x, y);
	}

	// Replaced function
	Game_Map.prototype.tileId = function(x, y, z) {
		var mapZ = MultiLevelMapManager.getZFromXY(x, y);
		var cx = MultiLevelMapManager.getXFromXY(x, y);
		var cy = MultiLevelMapManager.getYFromXY(x, y);

	    var width = $dataMap.width;
	    var height = $dataMap.height;
	    if (!MultiLevelMapManager.getDataMapAtLevel(mapZ)) {
	    	return 0;
	    	console.log(x, y, z, cx, cy, mapZ);
	    }
	    return MultiLevelMapManager.getDataMapAtLevel(mapZ).data[(z * height + cy) * width + cx] || 0;
	}

	// Reused function
	_old_Game_Map_prototype_updateParallax = Game_Map.prototype.updateParallax;
	Game_Map.prototype.updateParallax = function() {
		for (var z in this._layerData) {
			if (this._layerData.hasOwnProperty(z)) {
				var layerData = this._layerData[z] ;
			    if (layerData.parallaxLoopX) {
			        layerData.parallaxX = this.layerData._parallaxX + layerData.parallaxSx / this.tileWidth() / 2;
			    }
			    if (layerData.parallaxLoopY) {
			        layerData.parallaxY = this.layerData._parallaxY + layerData.parallaxSy / this.tileHeight() / 2;
			    }
			}
		}

		_old_Game_Map_prototype_updateParallax.call(this);
	}

	// Replaced function
	Game_Map.prototype.changeTileset = function(tilesetId, z) {
		if (z && Math.floor(z) != 0) {
			this._layerData[Math.floor(z)].tilesetId = tilesetId;
		} else {
	    	this._tilesetId = tilesetId;
	    }
	    this.refresh();
	}

	// Reused function
	old_Game_Map_prototype_changeBattleback = Game_Map.prototype.changeBattleback
	Game_Map.prototype.changeBattleback = function(battleback1Name, battleback2Name, z) {
		if (z && Math.floor(z) != 0) {
			this._layerData[Math.floor(z)].battleback1Name = battleback1Name;
	    	this._layerData[Math.floor(z)].battleback2Name = battleback2Name;
		} else {
			old_Game_Map_prototype_changeBattleback.apply(this, arguments);
	    	this._battleback1Name = battleback1Name;
	    	this._battleback2Name = battleback2Name;
	    }
	}

	// Reused function
	old_Game_Map_prototype_changeParallax = Game_Map.prototype.changeParallax;
	Game_Map.prototype.changeParallax = function(name, loopX, loopY, sx, sy, z) {
		if (z && Math.floor(z) != 0) {
			var layerData = this._layerData[Math.floor(z)];
		    layerData.parallaxName = name;
		    layerData.parallaxZero = ImageManager.isZeroParallax(layerData.parallaxName);
		    if (layerData.parallaxLoopX && !loopX) {
		        layerData.parallaxX = 0;
		    }
		    if (layerData.parallaxLoopY && !loopY) {
		        layerData.parallaxY = 0;
		    }
		    layerData.parallaxLoopX = loopX;
		    layerData.parallaxLoopY = loopY;
		    layerData.parallaxSx = sx;
		    layerData.parallaxSy = sy;
	    } else {
	    	old_Game_Map_prototype_changeParallax.apply(this, arguments);
	    }
	}

	// Reused function
	old_Game_Map_prototype_unlockEvent = Game_Map.prototype.unlockEvent;
	Game_Map.prototype.unlockEvent = function(eventId, z) {
		old_Game_Map_prototype_unlockEvent.call(this, MultiLevelMapManager.eventIdOffset(z) + eventId);
	}

	// Reused function
	_old_Game_Map_prototype_updateEvents = Game_Map.prototype.updateEvents;
	Game_Map.prototype.updateEvents = function() {
		if (!MultiLevelMapManager.isMapSetupDone()) return;
	    _old_Game_Map_prototype_updateEvents.call(this);
	};

	// Reused function
	_old_Game_Map_prototype_updateVehicles = Game_Map.prototype.updateVehicles;
	Game_Map.prototype.updateVehicles = function() {
		if (!MultiLevelMapManager.isMapSetupDone()) return;
	    _old_Game_Map_prototype_updateVehicles.call(this);
	};











	// **************************************************************************
	//
	//                           Game_CharacterBase
	//
	// **************************************************************************

	Object.defineProperties(Game_CharacterBase.prototype, {
		  z: 		{ get: function() { return this._z; }},
		  homeZ: 	{ get: function() { return this._homeZ; }},
		  relX:		{ get: function() { return MultiLevelMapManager.getXFromXY(this.x, this.y); }},
		  relY:		{ get: function() { return MultiLevelMapManager.getYFromXY(this.x, this.y); }},
		  relRealX:		{ get: function() { return MultiLevelMapManager.getXFromXY(this._realX, this._realY); }},
		  relRealY:		{ get: function() { return MultiLevelMapManager.getYFromXY(this._realX, this._realY); }},
	});

	// Reused function
	_old_Game_CharacterBase_prototype_initialize = Game_CharacterBase.prototype.initialize;
	Game_CharacterBase.prototype.initialize = function() {
		if (typeof(this._homeZ) === 'undefined') this._homeZ = 0;
		if (typeof(this._z) === 'undefined') this._z = 0;
		this._vz = 0;
		this._regionCheckPrevX = 0;
		this._regionCheckPrevY = 0;
		this._gravityInfluence = 1;

		_old_Game_CharacterBase_prototype_initialize.apply(this, arguments);
	}

	// New function
	Game_CharacterBase.prototype.setHomeZ = function(z) {
		this._homeZ = z;
	}

	// New function
	Game_CharacterBase.prototype.setGravityInfluence = function(val) {
		this._gravityInfluence = val;
	}

	// New function
	Game_CharacterBase.prototype.setZ = function(z, x, y) {
		if (!MultiLevelMapManager.isValidZ(z)) return;

	    if (arguments.length <= 1) {
	    	x = this.x;
	    	y = this.y;
	    }

	    if (Math.floor(this._z) != Math.floor(z) || x != this.x || y != this.y) {
		    var ox = MultiLevelMapManager.xOffset(z);
		    var oy = MultiLevelMapManager.yOffset(z);
		    var cx = MultiLevelMapManager.getXFromXY(x, y);
		    var cy = MultiLevelMapManager.getYFromXY(x, y);

			this._z = z;
			var dx = this._realX - this._x;
	    	var dy = this._realY - this._y;
	    	this.setPosition(ox + cx, oy + cy);
	    	this._realX += dx;
	    	this._realY += dy;
	   	} else {
	   		this._z = z;
	   	}
	}

	// Reused function
	_old_Game_CharacterBase_prototype_locate = Game_CharacterBase.prototype.locate;
	Game_CharacterBase.prototype.locate = function(x, y, z) {
		if (arguments.length <= 2) {
			z = z || this.z;
		} else {
			if (!MultiLevelMapManager.isValidZ(z)) return;
			this._z = z;
		}
		var ox = MultiLevelMapManager.xOffset(this.z);
	    var oy = MultiLevelMapManager.yOffset(this.z);

		_old_Game_CharacterBase_prototype_locate.call(this, ox + x, oy + y);
	}

	// Reused function
	old_Game_CharacterBase_prototype_update = Game_CharacterBase.prototype.update;
	Game_CharacterBase.prototype.update = function() {
		// Gravity stuff 
		if (this._gravityInfluence != 0 && Math.floor(this._z) != this._z) {
			this._vz = this._vz + Z_DESC_ACCEL * this._gravityInfluence / 60;
			var vzAcum = this._vz;

			while (vzAcum != 0 && Math.floor(this._z) != this._z) {
				if (this._gravityInfluence > 0) {
					var diff = this._z - Math.floor(this._z);
					if (diff > vzAcum) {
						this.setZ(this._z - vzAcum);
						vzAcum = 0;
					} else {
						if (this == $gamePlayer) {
							$gameParty.onPlayerWalk();
							this._followers.updateMove();
						}
						this.setZ(Math.floor(this._z));
						vzAcum -= diff;
					}
				} else if (this._gravityInfluence < 0) {
					var diff = this._z - Math.ceil(this._z);
					if (diff < vzAcum) {
						this.setZ(this._z - vzAcum);
						vzAcum = 0;
					} else {
						if (this == $gamePlayer) {
							$gameParty.onPlayerWalk();
							this._followers.updateMove();
						}
						this.setZ(Math.ceil(this._z));
						vzAcum -= diff;
					}
				}
				this.checkMapLevelChange();
			}
		} else {
			this._vz = 0;
			this.checkMapLevelChange();
		}

		old_Game_CharacterBase_prototype_update.call(this);
	}

	// New function
	Game_CharacterBase.prototype.checkMapLevelChange = function() {
		var positionChanged = false;
		var rx = Math.round(this.relRealX);
		var ry = Math.round(this.relRealY);
		if ((this._regionCheckPrevX != rx || this._regionCheckPrevY != ry)) {
			this._regionCheckPrevX = rx;
			this._regionCheckPrevY = ry;
			positionChanged = true;
		}

		this.doRegionActionAt(Math.round(this._realX), Math.round(this._realY), positionChanged);
	}

	// New function
	Game_CharacterBase.prototype.doRegionActionAt = function(x, y, positionChanged) {
		var regionId = $gameMap.tileId(x, y, 5);
		if (!positionChanged) {
			if (regionId == REGION_HOLE) {
				this.setZ(this._z - Z_DESC_START_DIST * this._gravityInfluence);
			}
		} else {
			switch (regionId) {
				case REGION_UP:
					this.setZ(this._z + 1);
				break;
				case REGION_DOWN:
					this.setZ(this._z - 1);
				break;
				case REGION_HOLE:
					this.setZ(this._z - Z_DESC_START_DIST * this._gravityInfluence);
				break;
			}
		}
	}

	// Reused function
	old_Game_CharacterBase_prototype_screenY = Game_CharacterBase.prototype.screenY;
	Game_CharacterBase.prototype.screenY = function() {
		return old_Game_CharacterBase_prototype_screenY.call(this) - this.z * $gameMap.tileHeight();
	}






	// **************************************************************************
	//
	//                                  Game_Event
	//
	// **************************************************************************

	// Reused function
	_old_Game_Event_prototype_initialize = Game_Event.prototype.initialize;
	Game_Event.prototype.initialize = function(mapId, eventId) {
		this._homeZ = MultiLevelMapManager.getZByMapId(mapId);
		this._z = this._homeZ;

		this._originalEventId = eventId - MultiLevelMapManager.eventIdOffset(this.z);

		_old_Game_Event_prototype_initialize.apply(this, arguments);
	}

	// New function
	Game_Event.prototype.mapId = function() {
		return this._mapId;
	}

	// New function
	Game_Event.prototype.originalEventId = function() {
		return this._originalEventId;
	}

	// Replaced function, but actually does the same
	Game_Event.prototype.page = function() {
		if (!this.event()) {
			console.log(this, ':(');
		}
		return this.event().pages[this._pageIndex];
	};

	// Replaced function
	Game_Event.prototype.event = function() {
		var off = MultiLevelMapManager.eventIdOffset(this._homeZ);
	    return MultiLevelMapManager.getDataMapAtLevel(this._homeZ).events[this._eventId - off];
	}

	// Replaced function, uses the originalEventId instead of the eventId
	Game_Event.prototype.meetsConditions = function(page) {
		var c = page.conditions;
		 if (c.switch1Valid) {
	        if (!$gameSwitches.value(c.switch1Id)) {
	            return false;
	        }
	    }
	    if (c.switch2Valid) {
	        if (!$gameSwitches.value(c.switch2Id)) {
	            return false;
	        }
	    }
	    if (c.variableValid) {
	        if ($gameVariables.value(c.variableId) < c.variableValue) {
	            return false;
	        }
	    }
	    if (c.selfSwitchValid) {
	        var key = [this._mapId, this._originalEventId, c.selfSwitchCh];
	        if ($gameSelfSwitches.value(key) !== true) {
	            return false;
	        }
	    }
	    if (c.itemValid) {
	        var item = $dataItems[c.itemId];
	        if (!$gameParty.hasItem(item)) {
	            return false;
	        }
	    }
	    if (c.actorValid) {
	        var actor = $gameActors.actor(c.actorId);
	        if (!$gameParty.members().contains(actor)) {
	            return false;
	        }
	    }
	    
	    return true;
	};









	// **************************************************************************
	//
	//                             Game_Interpreter
	//
	// **************************************************************************

	Object.defineProperties(Game_Interpreter.prototype, {
		  z: 		{ get: function() { return this._z; }},
	});

	// Reused function
	_old_Game_Interpreter_prototype_initialize = Game_Interpreter.prototype.initialize;
	Game_Interpreter.prototype.initialize = function(depth) {
		this._z = 0;
		this._originalEventId = 0;
		_old_Game_Interpreter_prototype_initialize.apply(this, arguments);
	}

	// Reused function
	_old_Game_Interpreter_prototype_setup = Game_Interpreter.prototype.setup;
	Game_Interpreter.prototype.setup = function(list, eventId) {
		this._z = MultiLevelMapManager.getZByEventId(eventId);
		_old_Game_Interpreter_prototype_setup.apply(this, arguments);

		this._mapId = MultiLevelMapManager.getMapIdByZ(this.z);
		this._originalEventId = eventId - MultiLevelMapManager.eventIdOffset(this.z);
	}

	// Replaced function
	Game_Interpreter.prototype.isOnCurrentMap = function() {
	    return MultiLevelMapManager.hasMapId(this._mapId);
	}

	// Reused function
	_old_Game_Interpreter_prototype_character = Game_Interpreter.prototype.character;
	Game_Interpreter.prototype.character = function(param, z) {
		var evIdOffset = MultiLevelMapManager.eventIdOffset((arguments.length <= 1) ? this.z : z)
		return _old_Game_Interpreter_prototype_character.call(this, (param > 0) ? (evIdOffset + param) : param)
	}

	// Reused function, uses the internal mapId instead of the $gameMap.mapId()
	_old_Game_Interpreter_prototype_gameDataOperand = Game_Interpreter.prototype.gameDataOperand;
	Game_Interpreter.prototype.gameDataOperand = function(type, param1, param2) {
		if (this._mapId) {
		    if (type == 7) { // Other
		        if (param1 == 0) { // Map ID
		            return this._mapId;
		       }
		    }
		}

	    return _old_Game_Interpreter_prototype_gameDataOperand.apply(this, arguments);
	}

	// Reused function, uses the originalEventId instead of the eventId
	// Conditional Branch
	_old_Game_Interpreter_prototype_command111 = Game_Interpreter.prototype.command111;
	Game_Interpreter.prototype.command111 = function() {
	    var result = false;
	    if (this._params[0] == 2) {
	    	// Self Switch
	        if (this._originalEventId > 0) {
	            var key = [this._mapId, this._originalEventId, this._params[1]];
	            result = ($gameSelfSwitches.value(key) === (this._params[2] === 0));
	        }

	        this._branch[this._indent] = result;
		    if (this._branch[this._indent] === false) {
		        this.skipBranch();
		    }
		    return true;
	   }
	   return _old_Game_Interpreter_prototype_command111.call(this);
	}

	// Reused function, uses the originalEventId instead of the eventId
	// Control Self Switch
	Game_Interpreter.prototype.command123 = function() {
	    if (this._originalEventId > 0) {
	        var key = [this._mapId, this._originalEventId, this._params[0]];
	        $gameSelfSwitches.setValue(key, this._params[1] === 0);
	    }
	    return true;
	}

	// Replaced function, calls using the internal value of Z
	// Change Map Name Display
	Game_Interpreter.prototype.command281 = function() {
	    if (this._params[0] === 0) {
	        $gameMap.enableNameDisplay(this.z);
	    } else {
	        $gameMap.disableNameDisplay(this.z);
	    }
	    return true;
	}

	// Replaced function, calls using the internal value of Z
	// Change Tileset
	Game_Interpreter.prototype.command282 = function() {
	    var tileset = $dataTilesets[this._params[0]];
	    for (var i = 0; i < tileset.tilesetNames.length; i++) {
	        ImageManager.loadTileset(tileset.tilesetNames[i]);
	    }
	    if (ImageManager.isReady()) {
	        $gameMap.changeTileset(this._params[0], this.z);
	        return true;
	    } else {
	        return false;
	    }
	}

	// Replaced function, calls using the internal value of Z
	// Change Battle Back
	Game_Interpreter.prototype.command283 = function() {
	    $gameMap.changeBattleback(this._params[0], this._params[1], this.z);
	    return true;
	}

	// Replaced function, calls using the internal value of Z
	// Change Parallax
	Game_Interpreter.prototype.command284 = function() {
	    $gameMap.changeParallax(this._params[0], this._params[1],
	        this._params[2], this._params[3], this._params[4], this.z);
	    return true;
	}

	// Replaced function, transform the X/Y to the virtual coordinates
	// Get Location Info
	Game_Interpreter.prototype.command285 = function() {
	    var x, y, value;
	    if (this._params[2] === 0) {  // Direct designation
	        x = this._params[3];
	        y = this._params[4];
	    } else {  // Designation with variables
	        x = $gameVariables.value(this._params[3]);
	        y = $gameVariables.value(this._params[4]);
	    }

	    x += MultiLevelMapManager.xOffset(this.z);
	    y += MultiLevelMapManager.yOffset(this.z);
	    switch (this._params[1]) {
	    case 0:     // Terrain Tag
	        value = $gameMap.terrainTag(x, y);
	        break;
	    case 1:     // Event ID
	        value = $gameMap.eventIdXy(x, y);
	        break;
	    case 2:     // Tile ID (Layer 1)
	    case 3:     // Tile ID (Layer 2)
	    case 4:     // Tile ID (Layer 3)
	    case 5:     // Tile ID (Layer 4)
	        value = $gameMap.tileId(x, y, this._params[1] - 2);
	        break;
	    default:    // Region ID
	        value = $gameMap.regionId(x, y);
	        break;
	    }
	    $gameVariables.setValue(this._params[0], value);
	    return true;
	}














	// **************************************************************************
	//
	//                                Game_Player
	//
	// **************************************************************************

	// Replaced function
	Game_Player.prototype.performTransfer = function() {
	    if (this.isTransferring()) {
	        this.setDirection(this._newDirection);
	        if (this._needsMapReload || !MultiLevelMapManager.hasMapId(this._newMapId)) {
	    		$gameMap.setup(this._newMapId);
	            this._needsMapReload = false;
	        } else {
	        	MultiLevelMapManager.mapSetupDone();
	        }
	    	this.locate(this._newX, this._newY);
	        this.refresh();
	        this.clearTransferInfo();
	    }
	}

	// Replaced function
	Game_Player.prototype.center = function(x, y) {
		var cx = MultiLevelMapManager.getXFromXY(x, y)
		var cy = MultiLevelMapManager.getYFromXY(x, y)
	    return $gameMap.setDisplayPos(cx - this.centerX(), cy - this.centerY());
	}







	// **************************************************************************
	//
	//                                Game_Vehicle
	//
	// **************************************************************************

	// Replaced function
	Game_Vehicle.prototype.loadSystemSettings = function() {
	    var vehicle = this.vehicle();
	    this._mapId = vehicle.startMapId;

	    
		var z = MultiLevelMapManager.getZByMapId(this._mapId);
		this.setZ(z, vehicle.startX, vehicle.startY);
	    this.setImage(vehicle.characterName, vehicle.characterIndex);
	}

	// Replaced function
	Game_Vehicle.prototype.setLocation = function(mapId, x, y) {
	    this._mapId = mapId;
	    
	    var z = MultiLevelMapManager.getZByMapId(this._mapId);
	    this.setZ(z, x, y);

	    this.refresh();
	}


	// **************************************************************************
	//
	//                                Game_Follower
	//
	// **************************************************************************

	// Replaced function
	Game_Follower.prototype.chaseCharacter = function(character) {
		var sz = character.z - this.z;
	    var sx = $gameMap.deltaX(this.relX, character.relX);
	    var sy = $gameMap.deltaX(this.relY, character.relY);
	    if (sx !== 0 && sy !== 0) {
	        this.moveDiagonally(sx > 0 ? 4 : 6, sy > 0 ? 8 : 2);
	    } else if (sx !== 0) {
	        this.moveStraight(sx > 0 ? 4 : 6);
	    } else if (sy !== 0) {
	        this.moveStraight(sy > 0 ? 8 : 2);
	    }
	    this.setMoveSpeed($gamePlayer.realMoveSpeed());
	}

	// Replaced function
	Game_Followers.prototype.synchronize = function(x, y, d) {
	    this.forEach(function(follower) {
	        follower.locate(x, y, $gamePlayer.z);
	        follower.setDirection(d);
	    }, this);
	}




	// **************************************************************************
	//
	//                                  Scene_Map
	//
	// **************************************************************************

	// Replaced function
	Scene_Map.prototype.onMapLoaded = function() {
	    if (this._transfer) {
	        $gamePlayer.performTransfer();
	    } else {
	    	MultiLevelMapManager.mapSetupDone();
	    }
	    this.createDisplayObjects();
	};





	// **************************************************************************
	//
	//                              Sprite_Character
	//
	// **************************************************************************

	// Replaced function
	Sprite_Character.prototype.tilesetBitmap = function(tileId) {
	    var tileset = $gameMap.tileset(this._character.z);
	    var setNumber = 5 + Math.floor(tileId / 256);
	    return ImageManager.loadTileset(tileset.tilesetNames[setNumber]);
	}

	// Replaced function
	Sprite_Character.prototype.updateBitmap = function() {
	    if (this.isImageChanged()) {
	        this._tilesetId = $gameMap.tilesetId(this._character.homeZ);
	        this._tileId = this._character.tileId();
	        this._characterName = this._character.characterName();
	        this._characterIndex = this._character.characterIndex();
	        if (this._tileId > 0) {
	            this.setTileBitmap();
	        } else {
	            this.setCharacterBitmap();
	        }
	    }
	}

	// Replaced function
	Sprite_Character.prototype.isImageChanged = function() {
	    return (this._tilesetId !== $gameMap.tilesetId(this._character.homeZ) ||
	            this._tileId !== this._character.tileId() ||
	            this._characterName !== this._character.characterName() ||
	            this._characterIndex !== this._character.characterIndex());
	}





	// **************************************************************************
	//
	//                              Spriteset_Map
	//
	// **************************************************************************

	// Reused function
	_old_Spriteset_Map_prototype_initialize = Spriteset_Map.prototype.initialize;
	Spriteset_Map.prototype.initialize = function(mapZ) {
		this._mapZ = mapZ;
		this._layerTilemap = {};
		this._layerTileset = {};
		this._layerParallax = {};
		this._layerParallaxName = {};

		_old_Spriteset_Map_prototype_initialize.call(this)
	}

	// Reused function
	_old_Spriteset_Map_prototype_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
	Spriteset_Map.prototype.createLowerLayer = function() {
		_old_Spriteset_Map_prototype_createLowerLayer.call(this);

		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0) {
				var index = self._baseSprite.children.indexOf(self._layerTilemap[z]);
				self._baseSprite.addChildAt(self._layerParallax[z], index);
			}
		})
	}

	// Reused function
	_old_Spriteset_Map_prototype_createParallax = Spriteset_Map.prototype.createParallax;
	Spriteset_Map.prototype.createParallax = function() {
		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0) {
			    var parallax = new TilingSprite();
			    parallax.move(0, 0, Graphics.width, Graphics.height);

			    self._layerParallaxName[z] = '';
			    self._layerParallax[z] = parallax;
			}
		});

		_old_Spriteset_Map_prototype_createParallax.call(this);
	}

	// Reused function
	_old_Spriteset_Map_prototype_createTilemap = Spriteset_Map.prototype.createTilemap;
	Spriteset_Map.prototype.createTilemap = function() {
		this._layerTilemap = {};

		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0) {
				var tilemap;
				if (Graphics.isWebGL()) {
			        tilemap = new ShaderTilemap();
			    } else {
			        tilemap = new Tilemap();
			    }
			    tilemap.tileWidth = $gameMap.tileWidth();
			    tilemap.tileHeight = $gameMap.tileHeight();
			    tilemap.setData($gameMap.width(), $gameMap.height(), MultiLevelMapManager.getDataMapAtLevel(z).data);
			    tilemap.horizontalWrap = $gameMap.isLoopHorizontal(z);
			    tilemap.verticalWrap = $gameMap.isLoopVertical(z);

			    self._layerTilemap[z] = tilemap;
		    }
	    });


	    MultiLevelMapManager.forAllLayers(function(z) {
			if (z < 0) self._baseSprite.addChild(self._layerTilemap[z]);
		})

	    _old_Spriteset_Map_prototype_createTilemap.call(this);

		MultiLevelMapManager.forAllLayers(function(z) {
			if (z > 0) self._baseSprite.addChild(self._layerTilemap[z]);
		})
	}

	// Reused function
	_old_Spriteset_Map_prototype_loadTileset = Spriteset_Map.prototype.loadTileset;
	Spriteset_Map.prototype.loadTileset = function(z) {
		if (arguments.length == 0) {
			this._layerTileset = {};

			var self = this;
			MultiLevelMapManager.forAllLayers(function(z) {
				//Includes z == 0
				self.loadZTileset(z);
		    });
	   } else {
	   		self.loadZTileset(0);
	   }
	}

	// New function
	Spriteset_Map.prototype.loadZTileset = function(z) {
		if (z && z != 0) {
			var tilemap = this._layerTilemap[z];
			var tileset = $gameMap.tileset(z);
		    if (tileset) {
		        var tilesetNames = tileset.tilesetNames;
		        for (var i = 0; i < tilesetNames.length; i++) {
		            tilemap.bitmaps[i] = ImageManager.loadTileset(tilesetNames[i]);
		        }
		        var newTilesetFlags = $gameMap.tilesetFlags(z);
		        tilemap.refreshTileset();
		        if (!tilemap.flags.equals(newTilesetFlags)) {
		            tilemap.refresh();
		        }
		        tilemap.flags = newTilesetFlags;
		    }

		    this._layerTileset[z] = tileset;
	    } else {
	    	_old_Spriteset_Map_prototype_loadTileset.call(this);
	    }
	}

	// New function
	Spriteset_Map.prototype.tilemap = function(z) {
		if (z && Math.floor(z) != 0)	return this._layerTilemap[Math.floor(z)];
		return this._tilemap;
	}
	// New function
	Spriteset_Map.prototype.parallax = function(z) {
		if (z && Math.floor(z) != 0)	return this._layerParallax[Math.floor(z)];
		return this._parallax;
	}

	// New function
	Spriteset_Map.prototype.setValidSpriteParent = function(sprite, z) {
		var zTilemap = this.tilemap(z);
		if (sprite.parent && sprite.parent != zTilemap) {
			sprite.parent.removeChild(sprite);
			zTilemap.addChild(sprite);
		}
	}

	// Reused function
	_old_Spriteset_Map_prototype_update = Spriteset_Map.prototype.update;
	Spriteset_Map.prototype.update = function() {
		for (var i = 0; i < this._characterSprites.length; i++) {
			var sprite = this._characterSprites[i];
			if (sprite._character)
				this.setValidSpriteParent(sprite, sprite._character.z);
	    }

		_old_Spriteset_Map_prototype_update.call(this);
	}

	// Reused function
	_old_Spriteset_Map_prototype_createDestination = Spriteset_Map.prototype.createDestination;
	Spriteset_Map.prototype.createDestination = function() {
		_old_Spriteset_Map_prototype_createDestination.call(this);

	    this._destinationSprite.setMapZ(Math.floor($gamePlayer.z));
	    this.setValidSpriteParent(this._destinationSprite, $gamePlayer.z)
	}

	// Reused function
	_old_Spriteset_Map_prototype_updateTileset = Spriteset_Map.prototype.updateTileset;
	Spriteset_Map.prototype.updateTileset = function() {
		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0 && self._layerTilemap[z] !== $gameMap.tileset(z)) {
		        self.loadZTileset(z);
			}
		});

	    _old_Spriteset_Map_prototype_updateTileset.call(this);
	}

	// Reused function
	_old_Spriteset_Map_prototype__canvasReAddParallax = Spriteset_Map.prototype._canvasReAddParallax;
	Spriteset_Map.prototype._canvasReAddParallax = function(z) {
		if (z && z != 0) {
			var index = this._baseSprite.children.indexOf(this._layerParallax[z]);
		    this._baseSprite.removeChild(this._layerParallax[z]);
		    this._layerParallax[z] = new TilingSprite();
		    this._layerParallax[z].move(0, 0, Graphics.width, Graphics.height);
		    this._layerParallax[z].bitmap = ImageManager.loadParallax(this._layerParallaxName[z]);
		    this._baseSprite.addChildAt(this._layerParallax[z], index);
		} else {
			_old_Spriteset_Map_prototype__canvasReAddParallax.call(this);
		}
	};

	// Reused function
	_old_Spriteset_Map_prototype_updateParallax = Spriteset_Map.prototype.updateParallax;
	Spriteset_Map.prototype.updateParallax = function() {
		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0) {
				var parallaxName = self._layerParallaxName[z];
				var parallax = self._layerParallax[z];

				if (parallaxName !== $gameMap.parallaxName(z)) {
			        self._layerParallaxName[z] = $gameMap.parallaxName(z);
			        parallaxName = self._layerParallaxName[z];

			        if (parallax.bitmap && Graphics.isWebGL() != true) {
			            this._canvasReAddParallax(z);
			        } else {
			            parallax.bitmap = ImageManager.loadParallax(parallaxName);
			        }
			    }
			    if (parallax.bitmap) {
			        parallax.origin.x = $gameMap.parallaxOx();
			        parallax.origin.y = $gameMap.parallaxOy();
			    }

			    //parallax.visible = (z <= $gamePlayer.z);
			}
		});

		this._lastZ

		_old_Spriteset_Map_prototype_updateParallax.call(this);
		//this._parallax.visible = (0 <= $gamePlayer.z);
	}

	// Reused function
	_old_Spriteset_Map_prototype_updateTilemap = Spriteset_Map.prototype.updateTilemap;
	Spriteset_Map.prototype.updateTilemap = function() {
		var self = this;
		MultiLevelMapManager.forAllLayers(function(z) {
			if (z != 0) {
				var tilemap = self._layerTilemap[z];
				tilemap.origin.x = $gameMap.displayX() * $gameMap.tileWidth();
			    tilemap.origin.y = $gameMap.displayY() * $gameMap.tileHeight();

				tilemap.origin.y += z * $gameMap.tileHeight();
		    	//tilemap.y = -z * $gameMap.tileHeight();

		    	//tilemap.visible = (z <= $gamePlayer.z);
			}
		});

		_old_Spriteset_Map_prototype_updateTilemap.call(this);
	    //this._tilemap.visible = (0 <= $gamePlayer.z);
	}








	// **************************************************************************
	//
	//                            Sprite_Destination
	//
	// **************************************************************************

	// Reused function
	_old_Sprite_Destination_prototype_initialize = Sprite_Destination.prototype.initialize;
	Sprite_Destination.prototype.initialize = function(mapZ) {
	    this._mapZ = mapZ || 0;
	    _old_Sprite_Destination_prototype_initialize.call(this);
	}

	// New function
	Sprite_Destination.prototype.setMapZ = function(mapZ) {
		this._mapZ = mapZ;
	}

	// Reused function
	_old_Sprite_Destination_prototype_updatePosition = Sprite_Destination.prototype.updatePosition;
	Sprite_Destination.prototype.updatePosition = function() {
		_old_Sprite_Destination_prototype_updatePosition.call(this);
	    this.y -= this._mapZ * $gameMap.tileHeight(this._mapZ);
	}









	// **************************************************************************
	//
	//          Example of compatibility changes when user other plugin
	//
	// **************************************************************************
	//
	//   I made this work with Quasi's old QuasiMovement. As I'm not sure if 
	// I can share part of Quasi's plugin, I'll just describe the modifications:
	//
	//   * Replaced the calls to $gameMap.width() and $gameMap.height() with 
	// $gameMap.realWidth() and $gameMap.realHeight()
	//   * Rewrote these functions:
	//
	if (Imported.Quasi_Movement) {
		// Replacing a function from this plugin
		Game_CharacterBase.prototype.checkMapLevelChange = function() {
			var edge = this.collider().gridEdge();
		    var x1   = edge[0];
		    var x2   = edge[1];
		    var y1   = edge[2];
		    var y2   = edge[3];

		    var rx1  = MultiLevelMapManager.getXFromXY(rx1, ry1)
		    var rx2  = MultiLevelMapManager.getXFromXY(rx2, ry2)
		    var ry1  = MultiLevelMapManager.getYFromXY(rx1, ry1)
		    var ry2  = MultiLevelMapManager.getYFromXY(rx2, ry2)
		    
		    var sameRegion = true;
		    var regionId = $gameMap.tileId(x1, y1, 5);
		    for (var x = x1; sameRegion && x <= x2; x++) {
		    	for (var y = y1; sameRegion && y <= y2; y++) {
		    		if (regionId != $gameMap.tileId(x, y, 5)) {
		    			sameRegion = false;
		    		}
		    	}
		    }

		    if (sameRegion) {
		    	var positionChanged = false;
				var cx = this.cx() / QuasiMovement.tileSize, cy = this.cy() / QuasiMovement.tileSize;
				var rcx = Math.floor(MultiLevelMapManager.getXFromXY(cx, cy));
				var rcy = Math.floor(MultiLevelMapManager.getYFromXY(cx, cy));

				if (this._regionCheckPrevX != rcx || this._regionCheckPrevY != rcy) {
					this._regionCheckPrevX = rcx;
					this._regionCheckPrevY = rcy;
					positionChanged = true;
				}
				this.doRegionActionAt(Math.floor(cx), Math.floor(cy), positionChanged);
			}
		}

		// Replacing this one's from QuasiMovement
		Game_CharacterBase.prototype.valid = function(dir) {
		    var edge = this.collider(dir).gridEdge();
		    var x1   = edge[0];
		    var x2   = edge[1];
		    var y1   = edge[2];
		    var y2   = edge[3];

		    //Way too lazy, should perhaps do a for that goes through each pair of values between x1-x2 and y1-y2
		    return MultiLevelMapManager.isValid(x1, y1) && MultiLevelMapManager.isValid(x2, y2);
		}
	}
})();