﻿//=============================================================================
// KageDesu_RTBS.js
//=============================================================================
//V1. Build 1.

/*:
 * @plugindesc Real-time battle system
 * @author Pheonix KageDesu
 *
 *
 * @param Fast battle
 * @desc Allow fast battle mode in game? (1 - Yes, 0 - No)
 * @default 1
 *
 * @help
 *
 * Info and settings see in file 'KageDesu RTBS.docx'
 *
 * Plugin Command:
 *   PKD fastOn         # Turn on Fast Battle for next battles
 *   PKD fastOff        # Turn off Fast Battle for next battles
 */
 
 /*:ru
 * @plugindesc Боевая система в реальном времени
 * @author Pheonix KageDesu
 *
 * @param Fast battle
 * @desc Включить возможность провести быструю битву? (1 - Да, 0 - Нет)
 * @default 1
 *
 * @help
 *
 * Описание и настройку системы смотрите в файле 'KageDesu RTBS.docx'
 *
 * Plugin Command: (Команды доп. модуля)
 *   PKD fastOn         # Разрешить быструю битву для последующих боёв
 *   PKD fastOff        # Запретить быструю битву для последующих боёв
 */

 "use strict";
 {
	 
	//=============================================================================
	//MAIN
	//=============================================================================

	var parameters = PluginManager.parameters('KageDesu_RTBS');
	var pkd_UseFastBattle = parameters['Fast battle'] || 1;

	//------------------------------------------------------------------------------
	 	//Template
	 	//END Template
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//BattleSystem
		function BattleSystem_Consts() { //Don't change this constants if you don't know what they do
	 		throw new Error('This is a static class');
	 	}

	 	BattleSystem_Consts.PLAYER_WINDOW_W 	= 260;
	 	BattleSystem_Consts.PLAYER_WINDOW_H 	= 120;
	 	BattleSystem_Consts.ALLY_WINDOW_W 	 	= 124;

	 	BattleSystem_Consts.STRING_FAST_BATTLE   = ['Fast battle','Быстрый бой'];
	 	BattleSystem_Consts.STRING_BEGIN_BATTLE  = ['Battle begin!', 'Бой начинается!'];
	 	BattleSystem_Consts.STRING_TIMER_READY   = ['Action!', 'Действуй!'];
	 	BattleSystem_Consts.STRING_POPUP_EVADE   = ['Evade', 'Evade'];
	 	BattleSystem_Consts.STRING_POPUP_MISS    = ['Miss', 'Мимо'];
	 	BattleSystem_Consts.STRING_POPUP_FAIL    = ['Fail', 'Fail'];
	 	BattleSystem_Consts.STRING_POPUP_ABSORB  = ['Absorb', 'Поглощено'];
	 	BattleSystem_Consts.STRING_POPUP_IMMUNE  = ['Immune', 'Иммунитет'];
	 	BattleSystem_Consts.STRING_POPUP_WEAK    = ['Weak', 'Уязвимость'];
	 	BattleSystem_Consts.STRING_POPUP_SKILL   = ['Ready!', 'Готов!'];
	 	BattleSystem_Consts.STRING_WAIT_TURN	 = ['Wait for turn','Ожидание хода'];
	 	BattleSystem_Consts.STRING_ITEMS_HELP	 = ['For battle','Используемые в бою'];


	 	BattleSystem_Consts.MINIMUM_TURN_WAIT = 20;  //Minimum turn time in frames
	 	BattleSystem_Consts.BATTLE_TURN_STEP = 140;  //Чем больше, тем дольше ждать хода 
	 	BattleSystem_Consts.BATTLE_FAST_KOEF = 0.2;  // Koef * Time in Fast Battle
	 	BattleSystem_Consts.FRAMES = 60; //Frame rate (don't change)
	 	BattleSystem_Consts.FONT = 'VL-Gothic-Regular';
	 	BattleSystem_Consts.TURN = (3 * BattleSystem_Consts.FRAMES); //Frames on each turn
	 	//END BattleSystem
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//SDK
	 	function SDK() {
	 		throw new Error('This is a static class, you baka...');
	 	}
	 	
	 	SDK.times = function(n, method) {
	 		for(var i = 0; i<n; i++) {
	 			method(i);
	 		}
	 	}

	 	SDK._isRu = null; //Don't use this
	 	SDK.isRU = function()
	 	{
	 		if(this._isRu == null)
	 		{
	 			this._isRu = 0;
	 			let language = window.navigator.userLanguage || window.navigator.language;
				if(language){
					if(language == 'ru')
						this._isRu = 1;
				}
	 		}
	 		return this._isRu;
	 	}

	 	SDK.check = function(value, base)
	 	{
	 		if(base === undefined)
	 			base = true;

	 		if(value === undefined)
	 			return base;
	 		else
	 			return value;
	 	}

	 	SDK.rand = function(min, max)
	 	{
	 		return Math.round(Math.random() * (max - min)) + min;
	 	}

	 	SDK.drawIcon = function(x, y, iconSymbol, bitmap, size)
	 	{
	 		size = SDK.check(size, 24);
	 		let icon = ImageManager.getIconPKD(iconSymbol, size);
	 		icon.addLoadListener(function(){
	 			bitmap.blt(icon, 0, 0, size, size, x, y);
	 		});
	 	}

	 	SDK.toCX = function(width, sourceWidth)
	 	{
	 		sourceWidth = SDK.check(sourceWidth, Graphics.width);
	 		return ((sourceWidth / 2) - (width / 2));
	 	}

		Array.prototype.delete = function() {
		    var what, a = arguments, L = a.length, ax;
		    while (L && this.length) {
		        what = a[--L];
		        while ((ax = this.indexOf(what)) !== -1) {
		            this.splice(ax, 1);
		        }
		    }
		    return this;
		};

		Array.prototype.include = function(value) {
			return (this.indexOf(value) != -1);
		}

		Array.prototype.max = function() {
		  return Math.max.apply(null, this);
		};

		Array.prototype.min = function() {
		  return Math.min.apply(null, this);
		};

		Rectangle.prototype.toArr = function() {
			return [this.x, this.y, this.width, this.height];
		}

	 	//END SDK
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Game_Interpreter
	    var _Game_Interpreter_pluginCommand =
	            Game_Interpreter.prototype.pluginCommand;
	    Game_Interpreter.prototype.pluginCommand = function(command, args) {
	        _Game_Interpreter_pluginCommand.call(this, command, args);
	        if (command === 'PKD') {
	            switch (args[0]) {
	            case 'circle':
	                SceneManager.push(Test_Circle);
	                break;
	            //case 'bar':
	            //    SceneManager.push(Test_Bar);
	            //    break;
	            //case 'window':
	            //    SceneManager.push(Test_Window);
	            //    break;
	            //case 'battle':
	            //	BattleManager.setup(1,true,true);
	            //	SceneManager.push(Scene_Battle);
	            //	break;
	            case 'fastOn':
	            	BattleManagerRTBS.allowFastBattle(true);
	            	break;
	            case 'fastOff':
	            	BattleManagerRTBS.allowFastBattle(false);
	            	break;
	            }
	        }
	    };
		//END Game_Interpreter
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Input
 		let pkd_Input_onKeyDown = Input._onKeyDown;
 		Input._onKeyDown = function(event) {
 			pkd_Input_onKeyDown.call(this, event);
 			let bn = this.KeyMapperPKD[event.keyCode];
 			if(bn) {
 				this._currentState[bn] = true;
 			}
 		}

		let pkd_Input_onKeyUp = Input._onKeyUp;
 		Input._onKeyUp = function(event) {
 			pkd_Input_onKeyUp.call(this, event);
		    let bn = this.KeyMapperPKD[event.keyCode];
		    if (bn) {
		        this._currentState[bn] = false;
		    }
		};

		Input.KeyMapperPKD = {
			9: 'tab', //GP LB
			32: 'space', //GP RB
			49: '1',
			50: '2',
			51: '3',
			52: '4',
			65: 'a', //GP X
			68: 'd', //GP B
			83: 's', // GP A
			87: 'w' //GP Y
		}

		Input.IsCancel = function() {
			if(Input.isGamepad()) {
				return Input.isTriggered('pageup'); //GP LB
			} else {
				return (Input.isTriggered('cancel') || TouchInput.isCancelled());
			}
		}

		Input._isGamepad = undefined; //Don't use this
		Input.isGamepad = function()
		{
			if(this._isGamepad !== undefined)
			{
				return this._isGamepad;
			}

			this._isGamepad = false;
			if(navigator.getGamepads)
			{
				 let gamepads = navigator.getGamepads();
				 if(gamepads)
				 {
				 	for(let i = 0; i<gamepads.length; i++)
				 	{
				 		if(gamepads[i] !== undefined)
				 		{
				 			this._isGamepad = true;
				 			break;
				 		}
				 	}
				 }
			}

			return this._isGamepad;
		}
	 	//END Input
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//IKey
	 	function IKey() {
			throw new Error('This is a static class');
		}

		IKey.convertToX = function(value)
		{
			switch(value)
			{
				case 'menu':
				case 'w':
					return 'Y';
				case 'shift':
				case 'a':
					return 'X';
				case 'ok':
				case 's':
					return 'A';
				case 'cancel':
				case 'd':
					return 'B';
				case 'pageup':
				case 'tab':
					return 'LB';
				case 'pagedown':
				case 'space':
					return 'RB';
				default:
					return '';
			}
		}

		IKey.W = function()
		{
			if(Input.isGamepad())
				return 'menu';
			else
				return 'w';
		}

		IKey.A = function()
		{
			if(Input.isGamepad())
				return 'shift';
			else
				return 'a';
		}

		IKey.S = function()
		{
			if(Input.isGamepad())
				return 'ok';
			else
				return 's';
		}

		IKey.D = function()
		{
			if(Input.isGamepad())
				return 'cancel';
			else
				return 'd';
		}

		IKey.TAB = function()
		{
			if(Input.isGamepad())
				return 'pageup';
			else
				return 'tab';
		}

		IKey.SPACE = function()
		{
			if(Input.isGamepad())
				return 'pagedown';
			else
				return 'space';
		}
	 	//END IKey
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//ImageManager
	 	ImageManager.PKDPath = 'img/pkd_BattleSystem/';
	 	ImageManager._iconCache = {};

		ImageManager.loadPKD = function(filename) {
	    	return this.loadBitmap(ImageManager.PKDPath, filename, 0, false);
		};

		ImageManager.getButtonPKD = function(button, size) {
			if(size === undefined)
				size = 38;

			if(Input.isGamepad()) //GAMEPAD XBOX
			{
				let name = IKey.convertToX(button);
				let btn = Bitmap.load(ImageManager.PKDPath + name + '_' + size + '.png');
				return btn;
			}
			else //KEYBOARD
			{
				var btn = Bitmap.load(ImageManager.PKDPath + 'Button_' + size + '.png');
				btn.addLoadListener(function() {
					btn.drawText(button.toUpperCase(), 0, 0, size + 1, size - 2, 'center');
				});
				return btn;
			}
		};

		//return bitmap
		ImageManager.getIconPKD = function(iconSymbol, size, forceCopy) {
			if(iconSymbol == null)
				iconSymbol = 'null';
			var key = iconSymbol + ':' + size;
			if(!this._iconCache[key])
				{
				if(typeof iconSymbol == 'string')
				{
					var icon_bitmap = this.loadPKD('icon_' + iconSymbol);
					this._iconCache[key] = icon_bitmap;
				}
				else
				{
					var bitmap = ImageManager.loadSystem('IconSet');
				    var pw = Window_Base._iconWidth;
				    var ph = Window_Base._iconHeight;
				    var sx = iconSymbol % 16 * pw;
				    var sy = Math.floor(iconSymbol / 16) * ph;
				    var icon_bitmap = new Bitmap(size, size);
				    icon_bitmap.addLoadListener(function() {
				    	icon_bitmap.blt(bitmap, sx, sy, pw, ph, 0, 0, size, size);
				    });
				    if(forceCopy) //Force new bitmap
				    	return icon_bitmap;
				    this._iconCache[key] = icon_bitmap;
				}
			}
			return this._iconCache[key];
		}
		//END ImageManager
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//ColorPKD
	 	class ColorPKD
	 	{
	 		constructor(r, g, b, a) {

	 			if(a === undefined)
	 				a = 255;

	 			this._r = r;
	 			this._g = g;
	 			this._b = b;
	 			this._a = a;
	 			this._staticCss = null;
	 		}

	 		prepare()
	 		{
	 			this._staticCss = Utils.rgbToCssColor(this._r, this._g, this._b);
	 		}

	 		toCSS() {
	 			if(this._staticCss == null) {
	 				this._staticCss = Utils.rgbToCssColor(this._r, this._g, this._b);
	 			}
	 			return this._staticCss;
	 		}

	 		toArr() {
	 			return [this._r, this._g, this._b, this._a];
	 		}

			getLightestColor(lightLevel)
	 		{
	 			let bf = 0.3 * this._r + 0.59 * this._g + 0.11 * this._b;
	 			var p = 0;
	 			var newColor = [0,0,0,0];
	 			if ((bf - lightLevel) >= 0) {
	 				if (bf >= 0) 
						p = Math.abs(bf - lightLevel)/lightLevel;
					newColor = this.toArr().map(function(c){ return (c - p * c);  });
	 			}
	 			else
	 			{
	 				if(bf >= 0)
	 					p = (lightLevel - bf)/(255 - bf);
	 				newColor = this.toArr().map(function(c){ return [(255 - c)*p+c,255].min();  });
	 			}

	 			return new ColorPKD(newColor[0],newColor[1],newColor[2],newColor[3]);
	 		}

	 		//STATIC
	 		static NONE() { return new ColorPKD(0,0,0,0);}
	 		static BLACK() { return new ColorPKD(0,0,0,255);}
	 		static WHITE() { return new ColorPKD(255,255,255,255);}
	 		static RED() { return new ColorPKD(255,0,0,255);}
	 		static GREEN() { return new ColorPKD(0,255,0,255);}
	 		static BLUE() { return new ColorPKD(0,0,255,255);}
	 		static AQUA() {return new ColorPKD(128,255,255,255);}
	 		static MAGENTA() {return new ColorPKD(128,0,128,255);}
	 		static YELLOW() {return new ColorPKD(255,255,0,255);}
	 		static ORANGE() {return new ColorPKD(255,128,0,255);} 
	 	}

	 	ColorPKD.NONE_Arr = [0,0,0,0];

	 	function ColorCSS() {
	 		throw new Error('This is a static class');
	 	}
	 	ColorCSS.WHITE = 'rgb(255,255,255)';
	 	ColorCSS.RED = 'rgb(255,0,0)';
	 	ColorCSS.GREEN = 'rgb(0,255,0)';
	 	ColorCSS.BLUE = 'rgb(0,0,255)';
	 	ColorCSS.YELLOW = 'rgb(255,255,0)';
	 	ColorCSS.ORANGE = 'rgb(255,128,0)';
	 	ColorCSS.MAGENTA = 'rgb(128,0,128)';
	 	ColorCSS.AQUA = 'rgb(128,255,255)';

		//END ColorPKD
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Bitmap
		let pkd_Bitmap_fillRect = Bitmap.prototype.fillRect;
		Bitmap.prototype.fillRect = function(x, y, width, height, color) {
		    if(color instanceof ColorPKD) {
		    	var context = this._context;		
		    	context.save(); 
		    	this.paintOpacity = color._a;
		    	context.fillStyle = color.toCSS();	
		    	context.fillRect(x, y, width, height);
		   		context.restore();
		    	this._setDirty();
		    } else {
		    	pkd_Bitmap_fillRect.call(this, x, y, width, height, color);
		    }
		};

		Object.defineProperty(Bitmap.prototype, 'paintOpacity', {
		    get: function() {
		        return this._paintOpacity;
		    },
		    set: function(value) {
		      this._paintOpacity = value;
		      this._context.globalAlpha = this._paintOpacity / 255;
		    },
		    configurable: true
		});
		//END Bitmap
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Sprite
	 	let pkd_Sprite_setBlendColor = Sprite.prototype.setBlendColor;
		Sprite.prototype.setBlendColor = function(color) {
			if(color instanceof ColorPKD) {
				pkd_Sprite_setBlendColor.call(this, color.toArr());
			}
			else  {
				pkd_Sprite_setBlendColor.call(this, color);
			}
		};
		//END Sprite
	//------------------------------------------------------------------------------

	//=============================================================================
	//BATTLE SYSTEM
	//=============================================================================

	//------------------------------------------------------------------------------
	 	//BattleManager
	 	let pkd_BattleManager_setup = BattleManager.setup;
	 	BattleManager.setup = function(troopId, canEscape, canLose) {
	 		BattleManagerRTBS.setup(troopId, canEscape, canLose);
	 		pkd_BattleManager_setup.call(this, troopId, canEscape, canLose);
	 	}

	 	BattleManager.forceAction = function(battler) {
	 		BattleManagerRTBS.getProcess().setForceActionBattler(battler);
	 	}

	 	BattleManager.abort = function() {
	 		BattleManagerRTBS.abortBattle();
	 	}

	 	BattleManager.processEscape = function() {
		    $gameParty.removeBattleStates();
		    $gameParty.performEscape();
		    SoundManager.playEscape();
		    var success = this._preemptive ? true : (Math.random() < this._escapeRatio);
		    if (success) {
		        this.displayEscapeSuccessMessage();
		        this._escaped = true;
		        BattleManagerRTBS.abortBattle();
		    } else {
		        this.displayEscapeFailureMessage();
		        this._escapeRatio += 0.1;
		        $gameParty.clearActions();
		    }
		    return success;
		};


	 	BattleManager.isTurnEnd = function() {
	 		return BattleManagerRTBS.isTurn();
	 	}

	 	BattleManager.isActionForced = function() {
	 		return BattleManagerRTBS.getProcess().isActionForced();
	 	}
	 	//END BattleManager
	 //------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//BattleManagerRTBS
	 	function BattleManagerRTBS() {
	 		throw new Error('This is a static class');
	 	}

	 	BattleManagerRTBS.init = function() { //On game start
	 		this._fastBattleAllowed = true;
	 		var state = $dataStates[2]; //Guard
	 		if(state.iconIndex == 0) {
	 			state.autoRemovalTiming = 1;
	 			state.iconIndex = 81;
	 		}
	 	}

		BattleManagerRTBS.setup = function(troopId, canEscape, canLose) {
			this._battleTimer = 0;
			this._phase = 'init'; //'init','start','battle', finish';
    		this._fastBattle = false;
		};

		BattleManagerRTBS.setScene = function(scene)
		{
			this._scene = scene;
			this._scene.addChild(this._spriteset);
	 		let lrs = this.getLayers().getLayers();
		 		lrs.forEach(function(l) {
		 			this._scene.addChild(l);
	 		}.bind(this));
		}

		BattleManagerRTBS.setFastBattle = function()
		{
			this._fastBattle = true;
			this.getAllMembers().forEach(function(member)
			{
				member.setFastBattle();
			});
		}

		BattleManagerRTBS.allowFastBattle = function(isAllowed) {
			this._fastBattleAllowed = isAllowed;
		}

		BattleManagerRTBS.canUseFastBattle = function() {
			return this._fastBattleAllowed && (pkd_UseFastBattle > 0);
		}

		BattleManagerRTBS.getLayers = function()
		{
			return this._layers;
		}

		BattleManagerRTBS.getScene = function() {
			return this._scene;
		}

		BattleManagerRTBS.getPlayer = function()
		{
			return $gameParty.leader();
		}

		BattleManagerRTBS.getSpriteset = function()
		{
			return this._spriteset;
		}

		BattleManagerRTBS.getProcess = function()
		{
			return this._process;
		}

		BattleManagerRTBS.getAllMembers = function()
		{
			return  $gameParty.battleMembers().concat($gameTroop.members()); 
		}

		BattleManagerRTBS.getInterface = function()
		{
			return this._interface;
		}

		BattleManagerRTBS.getPlayerInput = function()
		{
			return this._scene.getPlayerInput();
		}

		BattleManagerRTBS.isFastBattle = function() 
		{
			return (this._fastBattle == true);
		}

		BattleManagerRTBS.isBattle = function()
		{
			return (this._phase == 'battle');
		}

		BattleManagerRTBS.isTurn = function() { //End of turn?
			return (this._battleTimer == BattleSystem_Consts.TURN);
		}

		BattleManagerRTBS.isBusy = function()
		{
			return $gameMessage.isBusy(); //$gameTroop.isEventRunning() ||  SceneManager.isSceneChanging()
		}

		BattleManagerRTBS.initBattle = function()
		{
			$gameSystem.onBattleStart();
			$gameParty.onBattleStart();
			$gameTroop.onBattleStart();
			this._prepare_battle();
			this._phase = 'start';
			BattleManager.displayStartMessages();
		}

		BattleManagerRTBS.checkBattleEnd = function()
		{
			if(this._phase && this._phase != 'final')
			{
				if($gameParty.isEmpty()) {
					this.abortBattle();
					return true;
				}
				if ($gameParty.isAllDead()) {
					this._on_battle_end();
					BattleManager.processDefeat();
					return true;
				} 
				if($gameTroop.isAllDead()){
					this._on_battle_end();
					BattleManager.processVictory();
					return true;
				}
			}

			return false;
		}

		BattleManagerRTBS.abortBattle = function()
		{
			this._on_battle_end();
			BattleManager.processAbort();
		}

		BattleManagerRTBS.update = function() {
		    switch (this._phase) {
		        case 'start':
		            this._update_on_start();
		            break;
		        case 'battle':
		            this._update_on_battle();
		            break;
		        case 'final':
		        	this._update_on_final();
		        	break;
		        case 'finish':
		            BattleManager.updateBattleEnd();
		            this._phase = null;
		            break;
		    }
		};

		BattleManagerRTBS.startBattle = function() {
			this._interface.getEnemy().createEnemyInfo();
			this._interface.getLog().createPopups();
			this._scene.battleStart();
			this._phase = 'battle';
			this._turn_start();
		}

		BattleManagerRTBS.updateForWait = function()
		{
			this._spriteset.update();
			this._interface.update();
		}

		BattleManagerRTBS.stop = function()
		{
			this._interface.terminate();
			this._layers.terminate(this._scene);
		}

		//PRIVATE
		BattleManagerRTBS._prepare_battle = function()
		{
			this._layers = new PKD_BattleSystem_Layers();
			this._spriteset = new Spriteset_Battle();
			this._interface = new PKD_BattleSystem_Interface(); 
			this._process = new PKD_BattleSystem_BattleProcess(this._interface.getLog());		
		}

		BattleManagerRTBS._update_on_start = function()
		{
			this._interface.update();
			this._process.processEvent();
		}

		BattleManagerRTBS._update_on_battle = function() {
			this._interface.update();
			this._spriteset.update();
			if(this._battleTimer < BattleSystem_Consts.TURN) {
				this._battleTimer += 1;
			}
			this.getAllMembers().forEach(function(member){
				member.updateInBattle();
			});
			if(this.isTurn()) {
				this._turn_end();
			}
		}

		BattleManagerRTBS._update_on_final = function() {
			this._interface.update();
			this._spriteset.update();
			//this.getLayers().battleLog.addChild(this._spriteset.getFieldSprite());
			this._phase = 'finish';
		}

		BattleManagerRTBS._turn_start = function() {
			this._battleTimer = 0;
			console.log("Battle: New Turn");
			$gameTroop.increaseTurn();
		}

		BattleManagerRTBS._turn_end = function() {
			this.getAllMembers().forEach(function(member){
				member.onTurnEnd();
			});
			this._scene.onTurnEnd();
			this._turn_start();
		}

		BattleManagerRTBS._on_battle_end = function() { //This calls when battle is finish
			//this.getAllMembers().forEach(function(member){
			//	member.onBattleEnd();
			//});
			this._interface.hide();
			this._phase = 'final';
			this._battleTimer = 0;
		}
	 	//END BattleManagerRTBS
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_PopInfoManager
		function PKD_PopInfoManager() {
	 		throw new Error('This is a static class');
	 	}

	 	PKD_PopInfoManager.makeDamagePopUp = function(user) {
	 		let result = user.result();

	 		if(result.hpDamage != 0) {
	 			let value = PKD_PopInfoManager.HP(result.hpDamage, result.critical);
	 	    	this._apply_pop_up(user, value);
	 		}

	 		if(result.mpDamage != 0) {
	 			let value = PKD_PopInfoManager.MP(result.mpDamage, result.critical);
	 			this._apply_pop_up(user, value);
	 		}

	 		if(result.tpDamage != 0) {
	 			let value = PKD_PopInfoManager.TP(result.tpDamage, result.critical);
	 			this._apply_pop_up(user, value);
	 		}
	 	}

	 	PKD_PopInfoManager.makeZeroDamagePopUp = function(user) {
	 		let result = user.result();
	 		let value = PKD_PopInfoManager.HP(0, result.critical);
	 		this._apply_pop_up(user, value);
	 	}

	 	PKD_PopInfoManager.makeDrainPopUp = function(user) { //user - who get drained HP
	 		let result = user.result();
 			if(result.hpDamage != 0) {
 				let value = PKD_PopInfoManager.HP(result.hpDamage, result.critical);
 				value.getFontSettings()[2] = true;
 				this._apply_pop_up(user, value);
 			}

 			if(result.mpDamage != 0) {
 				let value = PKD_PopInfoManager.MP(result.mpDamage, result.critical);
 				value.getFontSettings()[2] = true;
 				this._apply_pop_up(user, value);
 			}
	 	}

	 	PKD_PopInfoManager.makeStatePopUp = function(user, stateId, isErase) {
	 		let state = $dataStates[stateId];
	 		if(state.iconIndex == 0)
	 			return;
	 		if(state.id == user.deathStateId()) 
	 			return;
	 		let value = PKD_PopInfoManager.STATE((user.isEnemy() ? "" : state.name), state.iconIndex, isErase);
	 		this._apply_pop_up(user, value);
	 	}

	 	PKD_PopInfoManager.makeItemPopUp = function(user) {
	 		let result = user.result();
	 		if(!user.isAlive()) return;
	 		if(result.missed) {
	 			this._apply_pop_up(user, PKD_PopInfoManager.TEXT(BattleSystem_Consts.STRING_POPUP_MISS[SDK.isRU()]));
	 			return;
	 		}

	 		if(result.evaded) {
	 			this._apply_pop_up(user, PKD_PopInfoManager.TEXT(BattleSystem_Consts.STRING_POPUP_EVADE[SDK.isRU()]));
	 			return;
	 		}

	 		if(result.isHit() && !result.success) {
	 			this._apply_pop_up(user, PKD_PopInfoManager.TEXT(BattleSystem_Consts.STRING_POPUP_FAIL[SDK.isRU()]));
	 			return;
	 		}
	 	}

	 	PKD_PopInfoManager.makeBuffPopUp = function(user, paramId, isPositive) {
	 		if(!BattleManagerRTBS.isBattle()) return;
	 		if(!user.isAlive()) return;
	 		let paramName = user.isEnemy() ? "" : TextManager.param(paramId);
	 		var temp = isPositive ? 1 : -1;
	 		let iconIndex = user.buffIconIndex(temp, paramId);
	 		let value = PKD_PopInfoManager.BUFF(paramName, iconIndex, isPositive);
	 		if(!user.getInfoPops().include(value)) {
	 			this._apply_pop_up(user, value);
	 		}
	 	}

	 	PKD_PopInfoManager.makeSkillRechargePopUp = function(user, skillId) {
	 		if(!BattleManagerRTBS.isBattle()) return;
	 		if(!user.isAlive()) return;
	 		if(user.isEnemy()) return; //This is for ActorEnemy, in version 1 not develop yet
	 		let skill = $dataSkills[skillId];
	 		let value = PKD_PopInfoManager.SKILL(skill.name, skill.iconIndex);
	 		if(!user.getInfoPops().include(value)) {
	 			this._apply_pop_up(user, value);
	 		}
	 	}

	 	PKD_PopInfoManager.calcRate = function(rate) {
	 		this.text = "";
	 		/*if(rate > 1) {
	 			this.text = BattleSystem_Consts.STRING_POPUP_WEAK[SDK.isRU()];
	 		} else if(rate === 0) {
	 			this.text = BattleSystem_Consts.STRING_POPUP_IMMUNE[SDK.isRU()];
	 		} else if(rate < 1) {
	 			this.text = BattleSystem_Consts.STRING_POPUP_ABSORB[SDK.isRU()];
	 		} else {
	 			this.text = "Resist"; //What a hell?
	 		}*/ //Not yet, my friend...
	 	}

	 	//STATIC
	 	PKD_PopInfoManager.HP = function(value, critical) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		var color = ColorCSS.YELLOW;
	 		if(value < 0) {
	 			color = ColorCSS.GREEN;
	 			value = Math.abs(value);
	 		} else if(critical) {
	 			color = ColorCSS.RED;
	 			fontSettings[1] = 34;
	 		}

	 		let x = new PKD_Object_PopUp(value, color, null, fontSettings);
	 		x.setNumered();
	 		return x;
	 	}

	 	PKD_PopInfoManager.TP = function(value, critical) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		var color = ColorCSS.ORANGE;
	 		if(value < 0) {
	 			color = ColorCSS.GREEN;
	 			value = Math.abs(value);
	 		} else if(critical) {
	 			color = ColorCSS.RED;
	 			fontSettings[1] = 34;
	 		}

	 		value = value + " " +TextManager.tpA;
	 		let x = new PKD_Object_PopUp(value, color, null, fontSettings);
	 		x.setNumered();
	 		return x;
	 	}

	 	PKD_PopInfoManager.MP = function(value, critical) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		var color = ColorCSS.MAGENTA;
	 		if(value < 0) {
	 			color = ColorCSS.BLUE;
	 			value = Math.abs(value);
	 		} else if(critical) {
	 			color = ColorCSS.MAGENTA;
	 			color._r = 120;
	 			fontSettings[1] = 34;
	 		}

	 		let x = new PKD_Object_PopUp(value, color, null, fontSettings);
	 		x.setNumered();
	 		return x;
	 	}

	 	PKD_PopInfoManager.STATE = function(name, iconIndex, isErase) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		fontSettings[2] = true;

	 		var temp = isErase ? "- " : "+ ";
	 		fontSettings[0] = BattleSystem_Consts.FONT;
	 		return new PKD_Object_PopUp(temp + name, null, iconIndex, fontSettings);
	 	}

	 	PKD_PopInfoManager.BUFF = function(name, iconIndex, isPositive) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		fontSettings[2] = true;

	 		var color = isPositive ? ColorCSS.GREEN : ColorCSS.RED;
	 		fontSettings[0] = BattleSystem_Consts.FONT;
	 		return new PKD_Object_PopUp(name, color, iconIndex, fontSettings);
	 	}

	 	PKD_PopInfoManager.TEXT = function(text) {
	 		return new PKD_Object_PopUp(text);
	 	}

	 	PKD_PopInfoManager.TEXT_WITH_COLOR = function(text, color) {
	 		return new PKD_Object_PopUp(text, color);
	 	}

	 	PKD_PopInfoManager.SKILL = function(name, iconIndex) {
	 		var fontSettings = PKD_Object_PopUp.FONT_DEFAULT();
	 		fontSettings[2] = true;
	 		return new PKD_Object_PopUp(BattleSystem_Consts.STRING_POPUP_SKILL[SDK.isRU()], ColorCSS.GREEN, iconIndex, fontSettings);
	 	}

	 	//PRIVATE
	 	PKD_PopInfoManager._apply_pop_up = function(user, value) {
	 		/*if(this.text === undefined)
	 			this.text = "";
	 		if(this.text != "") {
	 			if(value.isNumered()) value.setExtraText(this.text);
	 			this.text = "";
	 		}*/ //chotto matte, watashi no tomodachi
	 		user.addInfoPop(value);
	 	}

		//END PKD_PopInfoManager
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_Layers
	 	class PKD_BattleSystem_Layers
	 	{
	 		constructor()
	 		{
	 			let w = BattleSystem_Consts.PLAYER_WINDOW_W + ((BattleSystem_Consts.ALLY_WINDOW_W + 2) * ($gameParty.battleMembers().length - 1));
	 			this.layer_player = new WindowLayer();
	 			this.layer_player.move(0, Graphics.height - BattleSystem_Consts.PLAYER_WINDOW_H, w, BattleSystem_Consts.PLAYER_WINDOW_H);
	 			this.layer_player.x = SDK.toCX(this.layer_player.width);

	 			this.layer_battleZone = new WindowLayer();
	 			this.layer_battleZone.move(0,0,Graphics.width,Graphics.height);

	 			this.layer_playerLog = new WindowLayer();
	 			this.layer_playerLog.move(this.layer_player.x, this.layer_player.y, this.layer_player.width, this.layer_player.height);

	 			this.layer_battleLog = new WindowLayer();
	 			this.layer_battleLog.move(0,0,Graphics.width,Graphics.height);

	 			this.layer_input = new WindowLayer();
	 			this.layer_input.move(0,0,Graphics.width,Graphics.height);
	 		}

	 		getLayers() //return layers in Z condition
	 		{
	 			return [this.layer_player, this.layer_battleZone, this.layer_playerLog, this.layer_battleLog, this.layer_input];
	 		}

	 		update()
	 		{
	 			this.layer_player.update();
	 		}

	 		terminate(scene)
	 		{
	 			let lrs = this.getLayers();
	 			lrs.forEach(function(l) {
	 				scene.removeChild(l);
	 			});
	 		}
	 	}

	 	Object.defineProperties(PKD_BattleSystem_Layers.prototype, {
			player : { get: function() {return this.layer_player; }, configurable: true },
			battleZone : { get: function() {return this.layer_battleZone; }, configurable: true },
			playerLog : { get: function() {return this.layer_playerLog; }, configurable: true },
			battleLog : { get: function() {return this.layer_battleLog; }, configurable: true },
			input : { get: function() {return this.layer_input; }, configurable: true }
		});
	 	//END PKD_BattleSystem_Layers
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_BattleSystem_Interface
		class PKD_BattleSystem_Interface
		{
			constructor()
			{
				this._playerInterface = new PKD_BattleSystem_PlayerInterface(BattleManagerRTBS.getPlayer());
				this._log = new PKD_BattleSystem_Log();
				this._enemy = new PKD_BattleSystem_EnemyInterface(BattleManagerRTBS.getLayers().battleZone);
				this._allyInterface = [];
				$gameParty.battleMembers().forEach(function(member) {
					if(!member.isPlayer()) {
						this._allyInterface.push(new PKD_BattleSystem_AllyInterface(member));
					}
				}.bind(this));
			}

			getLog()
			{
				return this._log;
			}

			getEnemy()
			{
				return this._enemy;
			}

			hide()
			{
				this._playerInterface.hide();
				this._log.hide();
				this._enemy.hide();
				this._allyInterface.forEach(function(item) {
					item.hide();
				});
			}

			terminate()
			{
				this._playerInterface.terminate();
				this._log.terminate();
				this._allyInterface.forEach(function(item) {
					item.terminate();
				});
			}

			update()
			{
				this._log.update();
				this._enemy.update();
				this._playerInterface.update();
				this._allyInterface.forEach(function(item) {
					item.update();
				});
			}
		}
		//END PKD_BattleSystem_Interface
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_BattleSystem_PlayerInterface
		class PKD_BattleSystem_PlayerInterface
		{
			constructor(battler)
			{
				this._back_layer = BattleManagerRTBS.getLayers().player;
				this._layer = BattleManagerRTBS.getLayers().playerLog;
				this._width = BattleSystem_Consts.PLAYER_WINDOW_W;
				this._height = BattleSystem_Consts.PLAYER_WINDOW_H;
				this._size_a = 104; //Размер левой части окна - максимум, начало второй части
				this._size_b = this._width - this._size_a - 40; //Начало средней части
				this._size_c = this._width - 30; //Начало последней части
				this._actor = battler;
				this._hidden = false;
				this._color = ColorPKD.NONE();
				this._color.prepare();
				this._create();
				this.hide();
			}

			isHided()
			{
				return this._hidden;
			}

			hide()
			{
				if(this._hidden)
					return;
				this._sprite_background.visible = false;
				this._sprite_info.visible = false;
				this._sprite_values.visible = false;
				this._statusBar.setDrawMode(false);
				this._rechargeBar.setDrawMode(false);
				this._readyBar.hide();
				this._hidden = true;
			}

			open()
			{
				if(!this._hidden)
					return;
				this._sprite_background.visible = true;
				this._sprite_info.visible = true;
				this._sprite_values.visible = true;
				this._statusBar.setDrawMode(true);
				this._rechargeBar.setDrawMode(true);
				this._readyBar.show();
				this._hidden = false;
			}

			update()
			{
				if(this._actor.sprite.visible == false)
				{
					this.hide();
				}
				else
				{
					if(this._actor.sprite.inHomePosition())
						this.open();
				}

				if(this._hidden)
					return;
				
				this._bar_wt.update();
				this._bar_hp.update();
				this._bar_mp.update();
				if(this._bar_tp)
					this._bar_tp.update();
			}

			terminate() {
				clearInterval(this._thread); 
			}

			//PRIVATE
			_create()
			{
				this._rechargeBar = new PKD_Object_RechargeBar(this._size_c, 4, this._height, this._layer, this._actor);
				this._statusBar = new PKD_Object_StatusBar(this._layer.x - 80,0, BattleManagerRTBS.getLayers().battleLog, this._actor);
				this._create_sprites();
				this._start_thread();
			}

			_create_sprites()
			{
				//Background
				var bitmap = new Bitmap(this._width, this._height);
				bitmap.fillRect(0,0,this._width,this._height, new ColorPKD(0,0,50,100));
				let color = ColorPKD.WHITE();
				color._a = 48;
				color.prepare();
				bitmap.fillRect(this._size_a, 2, 2, this._height - 4, color);
				bitmap.fillRect(this._size_c + 3, 2, 2, this._height - 4, color);
				this._sprite_background = new Sprite(bitmap);

				//Info
				let x = this._actor.sprite.screenX;
				let y = this._actor.sprite.screenY;
				bitmap = new Bitmap(100 , 32);
				bitmap.fillRect(0,0,100,26, new ColorPKD(0,0,0,80));
				bitmap.fontSize = 18;
				bitmap.drawText(this._actor.name(), 0, -3, 100, 24, 'center');
				this._sprite_info = new Sprite(bitmap);
				this._sprite_info.x = x;
				this._sprite_info.y = y;
				
				//Bars
				let b_w = 16
				let b_p = 2
				bitmap = new Bitmap(this._size_b, b_w * 6 + b_p * 6);
				this._bar_hp = new PKD_Object_BarHP(bitmap, this._actor);
				this._bar_hp.setPosition(0, 0, this._size_b - 10, b_w);
				this._bar_mp = new PKD_Object_BarMP(bitmap, this._actor);
				this._bar_mp.setPosition(0, b_w + b_p, this._size_b - 10, b_w);
				if($dataSystem.optDisplayTp) {
					this._bar_tp = new PKD_Object_BarTP(bitmap, this._actor);
					this._bar_tp.setPosition(0, (b_w * 2) + (b_p * 2), this._size_b - 10, b_w);
				} else {
					this._bar_tp = null;
				}

				this._readyBar = new PKD_Object_ReadyBar(this._size_a + 12, 62, this._size_b - 10, this._actor, this._layer);

				this._bar_wt = new PKD_Object_BarWT(bitmap, this._actor);
				this._bar_wt.setPosition(0, (b_w * 5) + (b_p * 5), this._size_b - 10, b_w);

				this._sprite_values = new Sprite(bitmap);
				this._sprite_values.x = this._size_a + 12;
				this._sprite_values.y = 4;

				this._back_layer.addChild(this._sprite_background);
				this._layer.addChild(this._sprite_info);
				this._layer.addChild(this._sprite_values);
			}

			_start_thread() {
				if(this._thread)
					this.terminate();
				this._thread = setInterval(function() {
						this._rechargeBar.update();
						this._statusBar.update();
						this._readyBar.update();
				}.bind(this), 200);
			}
		}
		//END PKD_BattleSystem_PlayerInterface
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_BattleSystem_AllyInterface
		class PKD_BattleSystem_AllyInterface
		{
			constructor(battler) {
				this._back_layer = BattleManagerRTBS.getLayers().player;
				this._layer = BattleManagerRTBS.getLayers().playerLog;
				this._width = BattleSystem_Consts.ALLY_WINDOW_W;
				this._height = BattleSystem_Consts.PLAYER_WINDOW_H;
				this._actor = battler;
				this._hidden = false;
				this._create();
				this.hide();
			}

			update()
			{
				if(this._actor.sprite.visible == false)
				{
					this.hide();
				}
				else
				{
					if(this._actor.sprite.inHomePosition())
						this.open();
				}
				this._bar_wt.update();
				this._bar_hp.update();
			}

			terminate() {
				clearInterval(this._thread); 
			}

			isHided()
			{
				return this._hidden;
			}

			hide() {
				if(this._hidden)
					return;
				this._statusBar.setDrawMode(false);
				this._sprite_background.visible = false;
				this._sprite_info.visible = false;
				this._sprite_values.visible = false;
				this._hidden = true;
			}

			open() {
				if(!this._hidden)
					return;
				this._statusBar.setDrawMode(true);
				this._sprite_background.visible = true;
				this._sprite_info.visible = true;
				this._sprite_values.visible = true;
				this._hidden = false;
			}

			//PRIVATE
			_create()
			{
				let l = BattleManagerRTBS.getLayers().battleLog;
				this._statusBar = new PKD_Object_StatusBar(this._layer.x + this._actor.sprite.screenX + 98, -2, l, this._actor);
				this._statusBar.setNoDrawTime();
				this._statusBar.setLimit(3);
				this._create_sprites();
				this._start_thread();
			}

			_create_sprites() {
				//Background
				var bitmap = new Bitmap(this._width, this._height);
				bitmap.fillRect(0,0,this._width,this._height, new ColorPKD(0,0,50,100));
				let color = ColorPKD.WHITE();
				color._a = 48;
				color.prepare();
				bitmap.fillRect(96, 2, 2, this._height - 4, color);
				this._sprite_background = new Sprite(bitmap);
				this._sprite_background.x = this._actor.sprite.screenX;

				//Info
				let x = this._actor.sprite.screenX;
				let y = this._actor.sprite.screenY;
				bitmap = new Bitmap(96 , 28);
				bitmap.fillRect(0,0,96,26, new ColorPKD(0,0,0,80));
				bitmap.fontSize = 18;
				bitmap.drawText(this._actor.name(), 0, 0, 96, 24, 'center');
				this._sprite_info = new Sprite(bitmap);
				this._sprite_info.x = x;
				this._sprite_info.y = y;

				//Bars
				let b_w = 16
				let b_p = 2
				bitmap = new Bitmap(96, b_w * 2 + b_p * 2);
				this._bar_hp = new PKD_Object_BarHPE(bitmap, this._actor);
				this._bar_hp.setPosition(0, 0, 96, b_w);

				this._bar_wt = new PKD_Object_BarWT(bitmap, this._actor);
				this._bar_wt.setMiniInfo();
				//this._bar_wt.setPosition(0, (b_w * 1) + (b_p * 1), 96, b_w);
				this._bar_wt.setPosition(0, b_w + b_p, 96, b_w);

				this._sprite_values = new Sprite(bitmap);
				this._sprite_values.x = this._actor.sprite.screenX;
				this._sprite_values.y = this._height - bitmap.height;

				this._back_layer.addChild(this._sprite_background);
				this._layer.addChild(this._sprite_info);
				this._layer.addChild(this._sprite_values);
			}

			_start_thread() {
				if(this._thread)
					this.terminate();
				this._thread = setInterval(function() {
						this._statusBar.update();
				}.bind(this), 200);
			}
		}
		//END PKD_BattleSystem_AllyInterface
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_BattleSystem_EnemyInterface
		class PKD_BattleSystem_EnemyInterface
		{
			constructor(layer)
			{
				this._items = [];
				this._layer = layer;
				this._isSelected = false;
			}

			isTargeting()
			{
				return (this._isSelected == true);
			}

			createEnemyInfo()
			{
				this._items = [];
				$gameTroop.members().forEach(function(member) {
					this.addEnemyInfo(member);
				}.bind(this));
			}

			addEnemyInfo(enemy)
			{
				var item = new PKD_Object_EnemyInfo(enemy, this._layer);
				this._items.push(item);
			}

			select(index, iconIndex) //Select Enemy
			{
				//console.log(index);
				this.diselectAllEnemies();
				this._items[index].select(true, iconIndex);
				this._isSelected = true;
				//console.log("Enemy ", index, " selectd with icon ", iconIndex);		
			}

			refreshEnemyInfo(index) {
				if(this._items[index])
					this._items[index]._refresh();
			}

			hide()
			{
				this._items.forEach(function(item) {
					item.hide();
				});
			}

			diselectAllEnemies()
			{
				//console.log("All enemies diselected");
				this._isSelected = false;
				this._items.forEach(function(item) {
					item.select(false, null);
				});
			}

			update()
			{
				this._items.forEach(function(item) {
					item.update();
				});
			}

			//PRIVATE
		}
		//END PKD_BattleSystem_EnemyInterface
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_BattleInput
	 	class PKD_BattleSystem_BattleInput
	 	{
	 		constructor()
	 		{
	 			//Detect Mode
	 			this._mode = 'off';
	 			this._detect_mode();

	 			if(this._mode == 'off') {
	 				BattleManagerRTBS.startBattle();
	 			} else {
	 				this._layer = BattleManagerRTBS.getLayers().input;
	 				this._create_circle();
	 				this._create_battle_header();
	 			}

	 			this._waitMenu = false; //Ждать основного круга битвы
	 			this._flag_wait = false; //Ждать закрытие круга
	 		}

	 		isOff() {
	 			return (this._mode == 'off');
	 		}

	 		isFull() {
	 			return (this._mode == 'full');
	 		}

	 		hide()
	 		{
	 			if(this.isOff())
	 				return;

	 			this._circle.close();
	 		}

	 		isOpen()
	 		{
	 			return this._circle.isOpen();
	 		}

	 		open()
	 		{
	 			if(this.isOff())
	 				return;

	 			this._circle.open();
	 		}

	 		activate()
	 		{
	 			this._waitMenu = false;
	 			this._flag_wait = false;
	 		}

	 		update()
	 		{
	 			if(this.isOff())
	 				return;

	 			this._circle.update();

	 			if(this._waitMenu) {
	 				if(this._circle.isOpen())
	 					this._circle.close();
	 				else {
	 					if(!this._flag_wait) {
	 						BattleManagerRTBS.getScene().getPlayerInput().activate();
	 						this._flag_wait = true;
	 					}
	 				}

	 			} else {
		 			if(BattleManagerRTBS.isBattle())
		 			{
		 				this._update_on_main();
		 			}
		 			else
		 				this._update_on_start();
	 			}
	 		}

	 		//PRIVATE
	 		_detect_mode()
	 		{
	 			/*if(BattleManagerRTBS.canUseFastBattle() && BattleManager.canEscape()) {
	 				this._mode = 'full';
	 				return;
	 			}

	 			if(BattleManagerRTBS.canUseFastBattle() && !BattleManager.canEscape()) {
	 				this._mode = 'FOnly';
	 				return;
	 			}

	 			if(!BattleManagerRTBS.canUseFastBattle() && BattleManager.canEscape()) {
	 				this._mode = 'EOnly';
	 				return;
	 			}*/ 

	 			//Пока так
	 			this._mode = 'full'; 
	 		}

	 		_create_circle()
	 		{
	 			if(this._mode == 'off')
	 				return; //No circle

	 			this._circle = new PKD_Object_InputCircle((Graphics.width / 2) - 60, BattleManagerRTBS.getSpriteset().getY() + 100, this._layer, false);

	 			switch(this._mode) {
	 				case 'full':
		 				this._circle.circle2.setIcons(['null','escape','battle','fast']); 
		 				this._circle.circle.setHelps(['',TextManager.escape,TextManager.fight,
		 								BattleSystem_Consts.STRING_FAST_BATTLE[SDK.isRU()]]);
		 				this._circle.hideSegment(0, true);
	 					break;

	 				case 'EOnly':
	 					this._circle.circle2.setIcons(['escape','null','battle','null']); 
	 					this._circle.circle.setHelps([TextManager.escape,'',TextManager.fight, '']);
		 				this._circle.hideSegment(1, true);	
		 				this._circle.hideSegment(3, true);	
	 					break;

	 				case 'FOnly':
	 					this._circle.circle2.setIcons(['fast','null','battle','null']); 
	 					this._circle.circle.setHelps([BattleSystem_Consts.STRING_FAST_BATTLE[SDK.isRU()],'',TextManager.fight, '']);
	 					this._circle.hideSegment(1, true);	
		 				this._circle.hideSegment(3, true);	
	 					break;
	 			}

	 			this._circle.showHelp();
	 			SDK.times(4, function(i) {
		 			this._circle.addListener(i, function() { 
		 					this._function_on_click(i);
		 				}.bind(this));
	 			}.bind(this));

	 			//TEMP For FULL mode only
	 			if(!BattleManager.canEscape())
	 				this._circle.disableIcon(1);
	 			if(!BattleManagerRTBS.canUseFastBattle())
	 				this._circle.disableIcon(3);
	 		}

	 		_update_on_start()
	 		{
	 			if(BattleManagerRTBS.isBusy())
	 			{
	 				this._circle.close();
	 			}
	 			else
	 			{
	 				if(!this._circle.isOpen())
	 					this._circle.open();

	 				//START
	 				if(Input.isTriggered(IKey.S()) || Input.isTriggered('ok')) 
	 				{	
	 					this._function_on_click_start(2);
	 				}

	 				if(Input.isTriggered(IKey.A())) 
	 				{
	 					this._function_on_click_start(3);
	 				}
	 				
	 				if(Input.isTriggered(IKey.D())) 
	 				{
	 					this._function_on_click_start(1);
	 				}

	 				if(Input.isTriggered(IKey.W())) 
	 				{
	 					this._function_on_click_start(0);
	 				}
	 			}
	 		}

	 		_update_on_main()
	 		{
	 			if(BattleManagerRTBS.isBusy())
	 			{
	 				this._circle.close();
	 			}
	 			else
	 			{
	 				if(!this._circle.isOpen())
	 					this._circle.open();
	 			}

	 			//CONTINUE
	 			if(Input.isTriggered(IKey.S()) || Input.isTriggered('ok')) 
	 			{	
	 				this._function_on_click_main(2);
	 			}

 				if(Input.isTriggered(IKey.D())) 
 				{
 					this._function_on_click_main(1);
 				}

 				if(Input.isTriggered(IKey.W())) 
 				{
 					this._function_on_click_main(0);
 				}

 				if(Input.isTriggered(IKey.A())) 
 				{
 					this._function_on_click_main(3);
 				}

 				//To Battle (like Continue)
 				//if(Input.isTriggered('cancel') || TouchInput.isCancelled()) {
 				if(Input.IsCancel()) {
					this._function_on_click_main(2);
				}

	 		}

	 		_function_on_click(index) {
	 			if(BattleManagerRTBS.isBattle()) {
	 				this._function_on_click_main(index);
	 			} else 
	 				this._function_on_click_start(index);
	 		}

	 		_function_on_click_start(index)
	 		{
	 			if(!this.isOpen())
	 				return;
	 			switch(index)
	 			{
	 				//Escape
	 				case 0:
	 					if(this._mode == 'EOnly') {
	 						this._on_escape(0); //ESCAPE
	 					} 
	 					else if(this._mode == 'FOnly') {
	 						this._on_fastBattle(0); //FAST
	 						return;
	 					}
	 					break;
	 				case 1:
		 				if(this.isFull()) {
		 					//this._circle.click(1);
		 					this._on_escape(1); //ESCAPE
		 				}
	 					break;
	 				//Battle
	 				case 2:
	 					this._circle.click(2);
	 					this._waitMenu = true;
	 					this._start();
	 					break;
	 				//Fast
	 				case 3:
		 				if(this.isFull()) {
		 					//this._circle.click(3);
		 					this._on_fastBattle(3);
		 				}
	 					break;

	 			}
	 		}

	 		_function_on_click_main(index)
	 		{
	 			if(!this.isOpen())
	 				return;
	 			switch(index)
	 			{
	 				//Escape
	 				case 0:
	 					if(this._mode == 'EOnly') {
	 						this._on_escape(0); //ESCAPE
	 					} 
	 					break;
	 				case 1:
	 					if(this.isFull()) {
		 					this._on_escape(1); //ESCAPE
		 				}
	 					break;
	 				//Battle
	 				case 2:
	 					this._circle.click(2);
	 					this._waitMenu = true;
	 					this._to_battle_input();
	 					break;
	 				//Fast
	 				case 3:
	 					//Not allowed!
	 					break;

	 			}
	 		}

	 		_start()
	 		{
	 			this._layer.removeChild(this._header);

	 			//Not allow use FB when battle begin
	 			if(this.isFull()) {
	 				this._circle.disableIcon(3); 
	 			} else if(this._mode == 'FOnly')
	 			{
	 				this._circle.disableIcon(0);
	 			}

	 			BattleManagerRTBS.startBattle();
	 			this._circle.close();
	 		}

	 		_create_battle_header()
	 		{
	 			let bitmap = new Bitmap(Graphics.width, 54);
	 			bitmap.addLoadListener(function() {
	 				bitmap.fontSize = 48;
	 				bitmap.drawText(BattleSystem_Consts.STRING_BEGIN_BATTLE[SDK.isRU()], 0, 0, Graphics.width, 54, 'center');
	 			});

	 			this._header = new Sprite(bitmap);
	 			this._header.y = 10;
	 			this._layer.addChild(this._header);
	 		}

	 		_to_battle_input()
	 		{
	 			this._waitMenu = true;	 			
	 		}

	 		_on_escape(index) {
	 			if(BattleManager.canEscape()) {
	 				this._circle.click(index); 
 					var result = BattleManager.processEscape();
 					if(result == false) {
 						if(!BattleManagerRTBS.isBattle()) {
 							this._start(); //Start battle, if NOT in it
 							this._waitMenu = true;
 						}
 						else {
 							this._to_battle_input();
 						}
	 				}
	 			}
	 		}

	 		_on_fastBattle(index) {
	 			if(BattleManagerRTBS.canUseFastBattle()){
	 				this._circle.click(index); 
	 				BattleManagerRTBS.setFastBattle(true);
	 				this._start();
	 				this._waitMenu = true;
	 			}
	 		}
	 	}
	 	//END PKD_BattleSystem_BattleInput
	//------------------------------------------------------------------------------	

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_PlayerInput
	 	class PKD_BattleSystem_PlayerInput
	 	{
	 		constructor(battler, layer)
	 		{
	 			this._phase = 'wait';
	 			this._battler = battler;
	 			this._flag_icon_disabled = false;
	 			this._flag_start_target = false;
	 			this._layer = layer;
	 			this._circle = new PKD_Object_InputCircle((Graphics.width / 2) - 60, BattleManagerRTBS.getSpriteset().getY() + 100, this._layer, false); 			
	 			this._target = new PKD_BattleSystem_TargetInput(this._battler);
	 			this._sCircle = new PKD_BattleSystem_ChoiseInput(this, this._battler, this._target);

	 			this._circle.setHelps([TextManager.item, TextManager.guard, TextManager.attack, TextManager.skill]);
	 			//this._circle.showHelp();
	 			//this._circle.hideHelp();

	 			this._tKey = IKey.SPACE();
	 			this._waitMenu = false; //Если открыт круг выбора действий битвы, то true
	 			this._flag_wait = false;

	 			SDK.times(4, function(i) {
		 			this._circle.addListener(i, function() { 
		 					this._function_on_click(i);
		 				}.bind(this));
	 			}.bind(this));
	 		}

	 		isOpen()
	 		{
	 			return this._circle.isOpen();
	 		}

	 		targetInput()
	 		{
	 			return this._target;
	 		}

	 		onTargetClick() //Outer target click by mouse (touch)
	 		{
	 			if(this._circle.isOpen()) {
	 				return;
	 			}

	 			if(this._phase == 'attack')
	 				this._on_target_ok();
	 			else {
	 				//Choise
	 				this._sCircle.onTargetClick();
	 			}
	 		}

	 		activate()
	 		{
	 			this._waitMenu = false;
	 			this._flag_wait = false;
	 		}

	 		hide()
	 		{
	 			this._phase = 'wait';
	 		}

	 		update()
	 		{
	 			this.updateBasic();
	 			if(this._waitMenu == false)
	 				this._update_circle();
	 			else {
	 				if(this._circle.isOpen())
	 					this._circle.close();
	 				else {
	 					if(this._flag_wait == false) {
		 					BattleManagerRTBS.getScene().getBattleInput().activate();
		 					this._flag_wait = true;
		 				}
	 				}
	 			}
	 		}

	 		updateForWait()
	 		{
				this.updateBasic();
	 		}

	 		updateBasic()
	 		{
	 			this._circle.update();
	 			this._sCircle.update();

	 			if(this._is_ready() && this._battler.isAlive())
	 			{
	 				if(this._phase == 'wait') {
	 					this._phase = 'main';
	 					this._battler.makeActions(); 
	 				}
	 			} else
	 			{
	 				this._phase = 'wait';
	 			}

	 			if(this._phase == 'wait')
	 			{
	 				this._close_target();
	 				if(this._circle.isOpen())
	 					this._circle.close(); 
	 				if(this._sCircle.isOpen())
	 					this._sCircle.close();
	 				return; 
	 			}

	 			if(this._phase == 'items')
	 			{
	 				if(!this._battler.canMove()) 
	 				{
	 					this._sCircle.close();
	 					this._phase = 'main';
	 					this._close_target();
	 				}
	 			}
		 	}

	 		refresh()
	 		{
	 			this._sCircle.refresh();
	 		}

	 		//PRIVATE
	 		_update_circle()
	 		{
	 			if(this._phase != 'wait')
	 				this._target.update();

	 			switch(this._phase)
	 			{
	 				case 'main':
	 					this._update_on_main();
	 					break;
	 				case 'attack':
	 					this._update_on_attack();
	 					break;
	 				case 'defense':
	 					this._update_on_defense();
	 					break;
	 				case 'items':
	 					this._update_on_items();
	 					break;
	 			}
	 		}

	 		_update_on_main()
	 		{
	 			if(!this._circle.isOpen())
	 				this._circle.open();

	 			if(this._battler.canMove())
	 			{
	 				this._circle.reset();
	 				this._flag_icon_disabled = false;
	 				if(!this._battler.hasItems()) 
	 					this._circle.disableIcon(0); 
	 				if(!this._battler.canUseSkills()) 
	 					this._circle.disableIcon(3); 
	 				if(!this._battler.canAttack())
	 					this._circle.disableIcon(2);
	 				if(this._circle.isOpen())
	 					this._update_on_main_buttons(); 
	 			} else {
	 				if(!this._flag_icon_disabled)
	 				{
	 					this._flag_icon_disabled = true;
	 					SDK.times(4, function(i) { this._circle.disableIcon(i); }.bind(this));
	 				}
	 			}
	 		}

	 		_update_on_main_buttons()
	 		{
	 			//ATTACK
	 			if(Input.isTriggered(IKey.S()))
	 			{
	 				this._on_attack();
	 			}
	 			//DEFENSE
	 			if(Input.isTriggered(IKey.D()))
	 			{
	 				this._on_defense();
	 			}

	 			//SKILL
	 			if(Input.isTriggered(IKey.A()))
	 			{
	 				this._on_skills();
	 			}

	 			//ITEMS
	 			if(Input.isTriggered(IKey.W()))
	 			{
	 				this._on_items();
	 			}

	 			if(Input.isPressed(this._tKey))
		 			this._circle.showHelp(); 
		 		else
		 			this._circle.hideHelp();

	 			//TO BATTLE MENU
	 			//if(Input.isTriggered('cancel') || TouchInput.isCancelled()) {
	 			if(Input.IsCancel()) {
	 				if(!BattleManagerRTBS.getScene().getBattleInput().isOff()) {
						this._waitMenu = true;				
						this._circle.close();
					}
				}
	 		}

	 		_update_on_defense()
	 		{
	 			var action = this._battler.inputtingAction();
	 			if(action) {
		 			action.setGuard();
		 			this._perform();
	 			}
	 		}

	 		_update_on_attack()
	 		{
	 			if(this._circle.isOpen())
	 				return;

	 			//TODO Если делать разные ID для атаки, то надо определять выбор цели\союзника\всех\себя

	 			//Если враг один, то атака сразу его, без выбора цели
	 			let temp = $gameTroop.aliveMembers();
	 			if(temp.length == 1) {
	 				this._target.setExternalTarget(temp[0]);
	 				this._on_target_ok();
	 				return;
	 			}

	 			if(!this._flag_start_target)
	 			{
	 				this._target.start();
	 				this._flag_start_target = true;
	 			}

	 			if(Input.isTriggered('ok'))
	 			{
	 				this._on_target_ok();
	 			}

	 			//if(Input.isTriggered('cancel') || TouchInput.isCancelled())
	 			if(Input.IsCancel())
	 			{
	 				this._perform_end();
	 			}
	 		}

	 		_update_on_items()
	 		{
	 			if(this._circle.isOpen())
	 				return;

	 			if(this._sCircle.isComplete())
	 			{
	 				if(!this._sCircle.isOpen())
	 					this._phase = 'wait';
	 			}
	 		}

	 		_perform()
	 		{
	 			BattleManagerRTBS.getProcess().makeBattleAction(this._battler);
	 			this._perform_end();
	 		}

	 		_perform_end()
	 		{
	 			this._circle.close();
	 			this._target.stop();
	 			this._flag_start_target = false;
	 			this._phase = 'wait';
	 		}

	 		_function_on_click(index) {
	 			if(!this._circle.isOpen())
	 				return;
	 			switch(index) {
	 				case 0:
	 				this._on_items();
	 				break;
	 				case 1:
	 				this._on_defense();
	 				break;
	 				case 2:
	 				this._on_attack();
	 				break;
	 				case 3:
	 				this._on_skills();
	 				break;
	 			}
	 		}

	 		_on_attack()
	 		{
				if(this._battler.canAttack())
 				{
 					this._phase = 'attack';
 					this._target.setTargetIcon('attack');
 					this._circle.click(2);
 					this._circle.close();
 				}
	 		}

	 		_on_defense()
	 		{
	 			this._phase = 'defense';
	 			this._circle.click(1);
	 			this._circle.close();
	 		}

	 		_on_skills()
	 		{
				if(this._battler.canUseSkills()) 
 				{
 					this._phase = 'items';
 					this._circle.click(3);
					this._circle.close();
					this._sCircle.openSkills();
 				}
	 		}

	 		_on_items()
	 		{
	 			if(this._battler.hasItems()) 
 				{
 					this._phase = 'items';
 					this._circle.click(0);
					this._circle.close();
					this._sCircle.openItems();
 				}
	 		}

	 		_on_target_ok()
	 		{
	 			let action = this._battler.inputtingAction();
	 			if(action) {
					action.setAttack();
	 				this._perform();
	 			} else {
	 				this._perform_end();
	 			}
	 		}

	 		_close_target()
	 		{
	 			if (this._flag_start_target) {
					this._flag_start_target = false
					this._target.stop();
				}
	 		}

	 		_is_ready()
	 		{
	 			return (this._battler.isReadyForAction() && BattleManagerRTBS.isBattle());
	 		}
	 	}
	 	//END PKD_BattleSystem_PlayerInput
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_TargetInput
	 	class PKD_BattleSystem_TargetInput
	 	{
	 		constructor(battler)
	 		{
	 			this._battler = battler;
	 			this._isTarget = false;
	 			this._index = {'ally' : -1, 'enemy' : -1};
				this._iconIndex = null;	 	
				this._target = null;		
				this.setEnemyMode();
	 		}

	 		setEnemyMode() 
	 		{
	 			this._mode = 'enemy';
	 			//this._modeD = false;
	 			this._aliveSize = $gameTroop.aliveMembers().length;
	 		}

	 		setAllyMode()
	 		{
	 			this._mode = 'ally';
	 			//this._modeD = false;
	 			this._aliveSize = $gameParty.battleMembers().length;
	 		}

	 		/*setAllyModeD() //For dead Ally
	 		{
	 			this._mode = 'ally';
	 			this._modeD = true;
	 			this._aliveSize = $gameParty.battleMembers().length;
	 		}*/

	 		isEnemyMode()
	 		{
	 			return (this._mode == 'enemy');
	 		}

	 		isTargeting() {
	 			return (this._isTarget == true);
	 		}

	 		isTargetingForEnemy() {
	 			return (this._isTarget == true && this.isEnemyMode());
	 		}

	 		setTargetIcon(iconIndex)
	 		{
	 			this._iconIndex = iconIndex;
	 		}

	 		setExternalTarget(target) //External select by touch
	 		{
	 			var index = this._get_index();
	 			index = target.index();
	 			let action = this._battler.inputtingAction();
	 			if(action)
	 				action.setTarget(index);
	 			this._set_index(index);
	 			//Not visual selection, action instantly
	 		}

	 		start()
	 		{
	 			this._isTarget = true;
	 			var index = this._get_index();
	 			if(index < 0)
	 				this._set_index(0); 

	 			if(this.isEnemyMode()) {
		 			if(this._aliveSize != $gameTroop.members().length) 
		 			{
		 				this._set_index(0); 
		 				this._aliveSize = $gameTroop.members().length;
		 			} 

		 			if(index >= $gameTroop.members().length) {
		 				this._set_index(0); 
		 			}
	 			} else 
	 			{	
	 				//if(this._modeD) {
	 				//	this._set_index(0); 
	 				//} else {
	 					if(this._aliveSize != $gameParty.battleMembers().length) 
			 			{
			 				this._set_index(0); 
			 				this._aliveSize = $gameParty.battleMembers().length;
			 			} 

			 			if(index >= $gameParty.battleMembers().length) {
			 				this._set_index(0); 
			 			}
	 				//}
	 			}


	 			this._set_target();
	 		}

	 		stop()
	 		{
	 			if(this.isEnemyMode())
	 				BattleManagerRTBS.getInterface().getEnemy().diselectAllEnemies();
	 			else {
	 				$gameParty.battleMembers().forEach(function(member) {
	 					member.deselect();
	 				});
	 			}

	 			this._isTarget = false;
	 			this.setEnemyMode();
	 		}

	 		update()
	 		{
	 			//console.log("caller is " + arguments.callee.caller.toString());
	 			if(this._isTarget)
	 			{
	 				if(Input.isTriggered('right')) {
	 					this._next_target();
	 				}
	 				if(Input.isTriggered('left')) {
	 					this._prev_target();
	 				}

	 				if(this._target && !this._target.isAlive() && !this._modeD) {
	 					this._next_target();
	 				}
	 			}
	 		}

	 		//PRIVATE

	 		_get_index()
	 		{
	 			return this._index[this._mode];
	 		}

	 		_set_index(index)
	 		{
	 			this._index[this._mode] = index;
	 		}

	 		_up_index(up)
	 		{
	 			var x = this._index[this._mode];
	 			this._index[this._mode] = x + up;
	 		}

	 		_set_target()
	 		{
	 			if(this.isEnemyMode())
	 				this._target = $gameTroop.members()[this._get_index()];
	 			else
	 				this._target = $gameParty.battleMembers()[this._get_index()];

	 			//diselect ally
	 			if(!this.isEnemyMode())
		 			$gameParty.battleMembers().forEach(function(member) {
		 					member.deselect();
		 			});

	 			if(this._target) {
	 				let action = this._battler.inputtingAction();
	 				if(action) {
	 					action.setTarget(this._target.index());
	 					if(this.isEnemyMode())
	 						BattleManagerRTBS.getInterface().getEnemy().select(this._target.index(), this._iconIndex);
	 					else {
	 						//console.log("Selected ALLY ", this._target.name());
	 						this._target.select();
	 					}
	 				}
	 			}
	 		}

	 		_next_target()
	 		{
	 			console.log('next target');
	 			this._up_index(1);
	 			var index = this._get_index();
	 			if(this.isEnemyMode()) {
	 				if(index >= $gameTroop.members().length) 
	 					this._set_index(0);
	 			}
	 			else {
	 				 if(index >= $gameParty.battleMembers().length) 
	 					this._set_index(0);
	 			}
	 			this._set_target();
	 		}

	 		_prev_target()
	 		{
	 			console.log('prev target');
	 			this._up_index(-1);
	 			var index = this._get_index();
	 			if(this.isEnemyMode()) {
		 			if(index < 0) 
		 				this._set_index($gameTroop.members().length - 1);
		 		}
		 		else { 
	 				 if(index < 0) 
	 					this._set_index($gameParty.battleMembers().length - 1);
	 			}
	 			this._set_target();
	 		}
	 	}
	 	//END PKD_BattleSystem_TargetInput
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_ChoiseInput
	 	class PKD_BattleSystem_ChoiseInput
	 	{
	 		constructor(parentInput, battler, targetInput)
	 		{
	 			this._parent = parentInput;
	 			this._target = targetInput;
	 			this._battler = battler;
	 			this._phase = 'wait';

	 			this._items = null; //Set of Items or Skills
	 			this._disabledIndexes = []; //Array of disabled indexes, for ex. item don't alow to use
	 			this._selectedItem = null;
	 			this._actionCompleteFlag = false;
	 			this._flag_start_target = false;
	 			this._type = 'items';

	 			let x = parentInput._circle.x;
	 			let y = parentInput._circle.ySmall;
	 			let layer = parentInput._layer;
	 			this._circle = new PKD_Object_InputCircleSmall(x,y,layer,false); 

	 			this._tKey = IKey.SPACE();


	 			SDK.times(4, function(i) {
		 			this._circle.addListener(i, function() { 
		 					if(this._circle.isOpen())
		 						this._on_item_select(i);
		 				}.bind(this));
	 			}.bind(this));
	 		}

	 		onTargetClick()
	 		{
	 			if(this._selectedItem != null)
	 				this._perform();
	 		}

	 		isOpen()
	 		{
	 			return this._circle.isOpen();
	 		}

	 		openItems()
	 		{
	 			this._type = 'items';
	 			this.open();
	 		}

	 		openSkills()
	 		{
	 			this._type = 'skills';
	 			this.open();
	 		}

	 		isComplete()
	 		{
	 			return (this._actionCompleteFlag == true);
	 		}

	 		open()
	 		{
	 			this._init_items();
	 			this._circle.open();
	 			this._actionCompleteFlag = false;
	 			this._phase = 'main';
	 		}

	 		close()
	 		{
	 			this._on_action_end();
	 		}

	 		update()
	 		{
	 			this._circle.update();
	 			this.refresh();
	 			switch(this._phase)
	 			{
	 				case 'wait':
	 				this._update_on_wait();
	 				break;
	 				case 'main':
	 				this._update_on_main();
	 				break;
	 				case 'action':
	 				this._update_on_action();
	 				break;
	 			}

	 			if(Input.isPressed(this._tKey))
		 			this._circle.showHelp(); 
		 		else
		 			this._circle.hideHelp();
	 		}

	 		refresh()
	 		{
	 			if(this._phase == 'wait')
	 				return;
	 			for(let i = 0; i< this._items.length; i++)
	 			{
	 				let item = this._items[i];
	 				if(item) {
	 					if(this._battler.canUse(item))
	 					{
	 						this._circle.disableIcon(i, false);
	 						this._disabledIndexes.delete(i);

	 					} else {
	 						this._circle.disableIcon(i, true);
	 						if(!this._disabledIndexes.include(i))
	 							this._disabledIndexes.push(i);
	 					}
	 				} 
	 			}
	 		}

	 		//PRIVATE
	 		_init_items() {
	 			this._disabledIndexes = [];
	 			this._circle.reset();
	 			if(this._type == 'skills') {
	 				this._items = this._battler.getBattleSkills();
	 			} else {
	 				this._items = this._battler.getBattleItems();
	 			}
	 			let iconsArray = ['null','null','null','null'];
	 			let helpsArray = ['','','',''];
	 			for(let i = 0; i<4; i++) {
	 				if(this._items[i]) {
		 				var index = this._items[i].iconIndex;
		 				if(index) {
		 					iconsArray[i] = index;
		 				} 
		 				helpsArray[i] = this._items[i].name;
	 				} else {
	 					this._disabledIndexes.push(i);
	 					this._circle.hideSegment(i, true);
	 				}
	 			}
	 			this._circle.setIcons(iconsArray);
	 			this._circle.setHelps(helpsArray);
	 			this.refresh();
	 		}

	 		_perform() {
	 			BattleManagerRTBS.getProcess().makeBattleAction(this._battler);
	 			this._on_action_end();
	 		}

	 		_on_action_end() {
	 			this._circle.close();
	 			this._selectedItem = null;
	 			this._phase = 'wait';
	 			this._actionCompleteFlag = true;
	 		}

	 		_on_item_select(index) {
	 			if(this._disabledIndexes.include(index))
	 				return;

	 			this._circle.diselectAllIcons();
	 			this._selectedItem = this._items[index];
	 			var inputAction = this._battler.inputtingAction();
	 			if(!inputAction) return;
	 			if(this._type == 'skills') {
	 				inputAction.setSkill(this._selectedItem.id);
	 				this._battler.setLastBattleSkill(this._selectedItem);
	 			} else {
	 				inputAction.setItem(this._selectedItem.id);
	 				$gameParty.setLastItem(this._selectedItem);
	 			}

	 			this._circle.select(index);
	 			this._target.setTargetIcon(this._selectedItem.iconIndex);
	 			//Select Item
	 			    if (!inputAction.needsSelection()) {
				        this._perform(); //Perform action instantly
				    } else if (inputAction.isForOpponent()) {
				    	this._target.setEnemyMode();
				        this._circle.close();
				        this._phase = 'action';
				    } else if (inputAction.isForFriend() || inputAction.isForDeadFriend()) {
				    	this._target.setAllyMode();
				    	this._circle.close();
				        this._phase = 'action';
				    } 
	 		}

	 		_update_on_wait()
	 		{
	 			if(this._flag_start_target) {
	 				this._flag_start_target = false;
	 				this._target.stop();
	 			}
	 			if(this._circle.isOpen())
	 				this._circle.close();
	 		}

	 		_update_on_main()
	 		{
	 			if(this._parent.isOpen()) {
	 				return; //Wait for parent close
	 			}
	 			if(!this._circle.isOpen())
	 				this._circle.open();
	 			this._update_on_main_buttons();
	 		}

	 		_update_on_main_buttons()
	 		{

	 			if(Input.isTriggered(IKey.W()))
	 			{
	 				this._on_item_select(0);
	 			}

	 			if(Input.isTriggered(IKey.D()))
	 			{
	 				this._on_item_select(1);
	 			}

	 			if(Input.isTriggered(IKey.S()))
	 			{
	 				this._on_item_select(2);
	 			}

	 			if(Input.isTriggered(IKey.A()))
	 			{
	 				this._on_item_select(3);
	 			}

	 			/*if(Input.isTriggered('ok'))
	 			{
	 				if(this._selectedItem != null)
	 					this._perform();
	 			}*/ //deprecated

	 			//if(Input.isTriggered('cancel') || TouchInput.isCancelled())
	 			if(Input.IsCancel())
	 			{
	 				this._on_action_end();
	 			}
	 		}

	 		_update_on_action()
	 		{
	 			if(this.isOpen())
	 				return;

	 			//Если враг один, то атака сразу его, без выбора цели
	 			if(this._check_fast_action())
	 			{
	 				return;
	 			}

	 			if(!this._flag_start_target) {
	 				this._target.start();
	 				this._flag_start_target = true;
	 			}

	 			if(Input.isTriggered('ok'))
	 			{
	 				if(this._selectedItem != null)
	 					this._perform();
	 			}

	 			//if(Input.isTriggered('cancel') || TouchInput.isCancelled())
	 			if(Input.IsCancel())
	 			{
	 				this._target.stop();
	 				this._flag_start_target = false;
	 				this._selectedItem = null;
	 				this._circle.diselectAllIcons();
	 				this.refresh();
	 				this._phase = 'main';
	 			}
	 		}

	 		_check_fast_action()
	 		{
	 			var action = this._battler.inputtingAction();

	 			if(!action)
	 			{
	 				return false;
	 			}

	 			if(action.isForUser()) {
	 				this._target.setExternalTarget($gameParty.leader());
	 				this._perform();
	 				return true;
	 			}

	 			if(action.isForFriend())
	 			{
	 				let temp = $gameParty.battleMembers();
	 				if(temp.length == 1) {
	 					this._target.setExternalTarget(temp[0]);
	 					this._perform();
	 					return true;
	 				}
	 			}

	 			if(action.isForOpponent()) {
		 			let temp = $gameTroop.aliveMembers();
		 			if(temp.length == 1) {
		 				this._target.setExternalTarget(temp[0]);
		 				this._perform();
		 				return true;
		 			}
	 			}

	 			return false;
	 		}
	 	}
	 	//END PKD_BattleSystem_ChoiseInput
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_Log
	 	class PKD_BattleSystem_Log
	 	{
	 		constructor() {
	 			 this._logWindow = new Window_BattleLog();
	 			 this._killInfo = new PKD_Object_KillInfo(10, 5, Graphics.width - 10, 4, BattleManagerRTBS.getLayers().battleLog);

	 			 this._battlersM = [];
	 			 this._battlersM2 = [];
	 			 this._enemiesM = [];

	 			 this._battlers = [];
	 			 this._enemies = [];

	 			 this._start_thread();
	 		}

	 		createPopups() {
	 			let members = BattleManagerRTBS.getAllMembers();
	 			members.forEach(function(member) {
	 				this._create_machines(member);
	 			}.bind(this));
	 		}

	 		showAnimation(subject, targets, animationId)
	 		{
	 			if(animationId < 0) {
	 				this._show_attack_animation(subject, targets);
	 			} else {
	 				this._logWindow.showNormalAnimation(targets, animationId, false);
	 			}
	 		}
	 		//TODO В GameAction обработку Rounded урона добавить!!!
	 		onActionResult(subject, target)
	 		{
	 			if(target.result().used) {
	 				this._on_damage(target);
	 				this._on_failure(target);
	 				if (target.result().isStatusAffected()) 
	 					this._on_affected_status(target);
	 				if(!target.isAlive()) {
	 					this._killInfo.push(subject, target);
	 				}
	 			}
	 		}

	 		onCounterAttack(target) {
	 			target.performCounter();
	 		}

	 		onMagicReflection(target) {
	 			target.performReflection();
	 		}

	 		onSubstitute(substitute, target) {
	 			substitute.performSubstitute(target);
	 		}

	 		update()
	 		{
	 			this._update_popups();
	 		}

	 		hide() {
	 			this._killInfo.hide();
	 			this._killPopUps();
	 		}

	 		show() {
	 			this._killInfo.show();
	 		}

	 		terminate() {
	 			clearInterval(this._thread); 
	 		}


	 		//PRIVATE

	 		_start_thread() {
	 			if(this._thread)
					this.terminate();
				this._thread = setInterval(function() {
						this._killInfo.update();
				}.bind(this), 300);
	 		}

	 		_show_attack_animation(subject, targets)
	 		{
	 			if(subject.isEnemy())
	 			{
	 				 SoundManager.playEnemyAttack();
	 			}
	 			this._logWindow.showActorAttackAnimation(subject, targets);
	 		}

	 		_on_damage(target)
	 		{
	 			if(target.result().missed) {
	 				
	 				if(target.result().physical) {
	 					target.performMiss();
	 				} else {
	 					this._on_failure();
	 				}

	 			} else if (target.result().evaded) {
	 				 if (target.result().physical) {
	 				 	target.performEvasion();
	 				 }
	 				 else {
	 				 	target.performMagicEvasion();
	 				 }
	 			} else {
	 				 //HP
	 				 if (target.result().hpAffected) {
				        if (target.result().hpDamage > 0 && !target.result().drain) {
				            target.performDamage();
				        }
				        if (target.result().hpDamage < 0) {
				            target.performRecovery(); 
				        }
				    }
				    //MP
				    if (target.isAlive() && target.result().mpDamage !== 0) {
				        if (target.result().mpDamage < 0) {
				            target.performRecovery();
				        }
				    }
				    //TP
				    if (target.isAlive() && target.result().tpDamage !== 0) {
				        if (target.result().tpDamage < 0) {
				            target.performRecovery();
				        }
				    }
	 			}
	 			PKD_PopInfoManager.makeItemPopUp(target);
	 		}

	 		_on_failure(target)
	 		{
	 			//Empty (Not used yet)
	 		}

	 		_on_affected_status(target) {
	 			let states = target.result().addedStateObjects();
	 			states.forEach(function(state) {
	 				var state_msg = target.isActor() ? state.message1 : state.message2;
	 				if(state.id === target.deathStateId())
	 					target.performCollapse();
	 				this._add_state_info(target, state_msg);
	 			}.bind(this));
	 		}

	 		_add_state_info(target, msg) {
	 			//Empty (Not used yet)
	 		}

	 		_update_popups() {
	 			if(this._battlers.length != 0) {
		 			for(let i = 0; i<this._battlers.length; i++) 
		 			{
		 				let items = this._battlers[i].getInfoPops();
		 				if(items.length != 0) {
		 					for(let j = 0; j<items.length; j++)
		 					{
		 						let item = items[j];
		 						if(item.hasIcon()) {
		 							this._battlersM2[i].push(item);
		 						} else {
		 							this._battlersM[i].push(item);
		 						}
		 					}
		 				}
		 				this._battlers[i].clearInfoPops();
		 			}
	 			}

	 			if(this._enemiesM.length != 0) {
	 				for(let i = 0; i<this._enemies.length; i++) 
		 			{
		 				let items = this._enemies[i].getInfoPops();
		 				if(items.length != 0) {
		 					for(let j = 0; j<items.length; j++) {
		 						this._enemiesM[i].push(items[j]);
		 					}
		 				}
		 				this._enemies[i].clearInfoPops();
		 			}
	 			}

	 			this._enemiesM.forEach(function(item) {
	 				item.update();
	 			});

	 			this._battlersM.forEach(function(item) {
	 				item.update();
	 			});

	 			this._battlersM2.forEach(function(item) {
	 				item.update();
	 			});

	 		}

	 		_killPopUps() {
	 			this._enemiesM.forEach(function(item) {
	 				item.clear();
	 			});

	 			this._battlersM.forEach(function(item) {
	 				item.clear();
	 			});

	 			this._battlersM2.forEach(function(item) {
	 				item.clear();
	 			});
	 		}

	 		_create_machines(battler) {
	 			if(!battler.isSpriteVisible()) return;
	 			if(battler.isPlayer()) {
	 				this._battlers.push(battler);
	 				this._battlersM.push(new PKD_Object_PopUpMachineA(battler, BattleManagerRTBS.getLayers().playerLog));
	 				this._create_machine_ext(0, BattleSystem_Consts.PLAYER_WINDOW_W);

	 			} else if(battler.isEnemy()) {
	 				this._enemies.push(battler);
	 				let x = battler.sprite.screenX;
	 				let y = battler.sprite.screenCY - 32;
	 				let width = battler.sprite.screenW;
	 				this._enemiesM.push(new PKD_Object_PopUpMachine(x,y, width, 4, BattleManagerRTBS.getLayers().battleZone));
	 			} else if(battler.isActor()) {
	 				//console.log("machine for ally");
	 				this._battlers.push(battler);
	 				this._battlersM.push(new PKD_Object_PopUpMachineA(battler, BattleManagerRTBS.getLayers().playerLog));
	 				this._create_machine_ext(battler.sprite.x, BattleSystem_Consts.ALLY_WINDOW_W);
	 			}
	 		}

	 		_create_machine_ext(xOffset, width)
	 		{
	 			let vp = BattleManagerRTBS.getLayers().player;
	 			let x = vp.x;
	 			let y = vp.y - 60;
	 			let l = BattleManagerRTBS.getLayers().battleLog;
	 			this._battlersM2.push(new PKD_Object_PopUpMachine(x + xOffset,y, width, 4, l));
	 		}

	 	}
	 	//END PKD_BattleSystem_Log
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_BattleSystem_BattleProcess
	 	class PKD_BattleSystem_BattleProcess
	 	{
	 		constructor(battleLog)
	 		{
	 			this._subject = null;
	 			this._action = null;
	 			this._forceAction = null;
	 			this._log = battleLog;
	 		}

	 		makeBattleAction(battler)
	 		{
	 			this._subject = battler;
	 			this._process_action()
	 		}

	 		//PRIVATE
	 		_process_action()
	 		{
	 			if(this._subject == null || !this._subject.currentAction())
	 			{
	 				return;
	 			}
	 			var action = this._subject.currentAction();
	 			if(action) {
	 				this._action = action;
	 				action.prepare();
	 				if(action.isValid()) {
	 					this._start_action();
	 					this._end_action();
	 				}
	 				this._subject.removeCurrentAction();
	 			}
	 		}

	 		_start_action()
	 		{
	 			if(!BattleManagerRTBS.isFastBattle()) {
	 				//this._subject.performActionStart(); This is motion only, not need
	 				this._subject.requestEffect('whiten');
	 			}

	 			//this._subject.performAction(); This is motion only, not need
    			var targets = this._action.makeTargets();
    			this._subject.useItem(this._action.item());
    			this._action.applyGlobal();
    			if(!BattleManagerRTBS.isFastBattle())
    				this._log.showAnimation(this._subject, targets, this._action.item().animationId);
				targets.forEach(function(target) {
					this._invoke_action(target, this._action);
				}.bind(this));
	 		}

	 		_end_action()
	 		{
	 			this._subject.performActionEnd();
	 			this._subject.onAllActionsEnd();
	 			BattleManagerRTBS.checkBattleEnd();
	 			this._subject.makeActions();
	 		}

	 		_invoke_action(target, action)
	 		{
			    if (Math.random() < action.itemCnt(target)) {
			        this.invokeCounterAttack(this._subject, target);
			    } else if (Math.random() < action.itemMrf(target)) {
			        this.invokeMagicReflection(this._subject, target);
			    } else {
			        this.invokeNormalAction(this._subject, target);
			    }
			    this._subject.setLastTarget(target);
	 		}

	 		invokeNormalAction(subject, target) {
			    var realTarget = this.applySubstitute(target);
			    this._action.apply(realTarget);
			    this._log.onActionResult(subject, target);
			};

			invokeCounterAttack(subject, target) {
				this._log.onCounterAttack(target);
			    var action = new Game_Action(target);
			    action.setAttack();
			    action.apply(subject);
			    this._log.onActionResult(subject, target);
			};

			invokeMagicReflection(subject, target) {
				this._log.onMagicReflection(target);
			    this._action.apply(subject);
			    this._log.onActionResult(subject, target);
			};

			applySubstitute(target) {
			    if (this.checkSubstitute(target)) {
			        var substitute = target.friendsUnit().substituteBattler();
			        if (substitute && target !== substitute) {
			            this._log.onSubstitute(substitute, target);
			            return substitute;
			        }
			    }
			    return target;
			};

			checkSubstitute(target) {
			    return target.isDying() && !this._action.isCertainHit();
			};

			isActionForced() {
			    return this._forceAction != null;
			};

			setForceActionBattler(battler) {
			    this._forceAction = battler;
			};

			processForcedAction() {
			    if (this.isActionForced()) {
			    	console.log("Process Forced!!!");
			    	var last_subject = this._subject;
			        this._subject = this._forceAction;
			        this._forceAction = null;
			        this._process_action();
			        this._subject = last_subject;
			    }
			};

			processEvent()
			{
				this.processForcedAction();
				$gameTroop.updateInterpreter();
				if($gameTroop.isEventRunning() || BattleManagerRTBS.checkBattleEnd())
				{
					return;
				}
   			    $gameTroop.setupBattleEvent();
   			    if ($gameTroop.isEventRunning() || SceneManager.isSceneChanging()) {
			        return;
			    }
			}
	 	}
	 	//END PKD_BattleSystem_BattleProcess
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_BattleSystem_SceneProcess
		class PKD_BattleSystem_SceneProcess 
		{
			constructor()
			{
				this.battleInput = new PKD_BattleSystem_BattleInput(BattleManagerRTBS.getLayers().input); 	
	 			this.playerInput = null;	
			}

			battleStart()
			{
				this.playerInput = new PKD_BattleSystem_PlayerInput(BattleManagerRTBS.getPlayer(), BattleManagerRTBS.getLayers().input);
	 			this._process_event();
			}

			update()
			{
				if(BattleManagerRTBS.isBusy()) {
					this.updateForWait();
				} else
					this.updateBasic();
			}

			updateForWait()
			{
				BattleManagerRTBS.updateForWait();
				this.hide();
				this.battleInput.update();
				if(this.playerInput)
					this.playerInput.updateForWait();
			}

			updateBasic()
			{
				this.battleInput.update();
				BattleManagerRTBS.update();
				if(BattleManagerRTBS.isBattle())
					this._update_battle();
				else
					this._update_start();
			}

			onTurnEnd()
	 		{
	 			this._process_event();
	 		}

	 		hide()
	 		{
	 			this.battleInput.hide();
	 			if(this.playerInput)
	 				this.playerInput.hide();
	 		}

			//PRIVATE
			_update_start()
	 		{
	 			if(!this.battleInput.isOpen())
	 				this.battleInput.open();
	 			this.battleInput.update();
	 		}

	 		_update_battle()
	 		{
	 			this._process_event();
	 			this._process_enemy();
	 			this._process_player();
	 			this._process_ally();
	 			BattleManagerRTBS.checkBattleEnd();
	 		}

	 		_process_player()
	 		{
	 			if(BattleManagerRTBS.isFastBattle()) {
	 				let p = BattleManagerRTBS.getPlayer();
	 				if(p.isReadyForAction() && p.isAlive()) {
	 					p.makeActions();
	 					BattleManagerRTBS.getProcess().makeBattleAction(p);
	 				}
	 			} 
	 			else
	 			{
	 				this.playerInput.update();
	 			}
	 		}

	 		_process_enemy()
	 		{
	 			$gameTroop.members().forEach(function(member){
	 				if(member.isReadyForAction() && member.isAlive()) {
	 					member.makeActions();
	 					BattleManagerRTBS.getProcess().makeBattleAction(member);
	 				}
	 			});
	 		}

	 		_process_ally()
	 		{
	 			$gameParty.battleMembers().forEach(function(member){
	 				if(member.isReadyForAction() && member.isAlive() && !member.isPlayer()) {
	 					member.makeActions();
	 					BattleManagerRTBS.getProcess().makeBattleAction(member);
	 				}
	 			});
	 		}

	 		_process_event()
	 		{
	 			/*while(!SceneManager.isSceneChanging()) {
	 				BattleManagerRTBS.waitForMessage();
	 				BattleManagerRTBS.getProcess().processEvent();
	 				if(!$gameTroop.isEventRunning())
	 					break;
	 				BattleManagerRTBS.updateForWait();
	 			}*/
	 			BattleManagerRTBS.getProcess().processEvent();
	 		}
		}	
		//END PKD_BattleSystem_SceneProcess
	//------------------------------------------------------------------------------

	//=============================================================================
	//SPRITES
	//=============================================================================

	//------------------------------------------------------------------------------
	 	//Spriteset_Battle
	 	Spriteset_Battle.prototype.getY = function() //NEW
	 	{
	 		return this._baseSprite.y;
	 	}

	 	/*Spriteset_Battle.prototype.getFieldSprite = function() //NEW
	 	{
	 		return this._fieldSprite;
	 	}*/

	 	Spriteset_Battle.prototype.createActors = function() {
		    this._actorSprites = [];
		    for (var i = 0; i < $gameParty.maxBattleMembers(); i++) {
		        this._actorSprites[i] = new Sprite_ActorPKD();
		        BattleManagerRTBS.getLayers().playerLog.addChild(this._actorSprites[i]);
		    }
		};

		/*let pkd_SpritesetBattle_createBattleback = Spriteset_Battle.prototype.createBattleback;
		Spriteset_Battle.prototype.createBattleback = function() {
		    pkd_SpritesetBattle_createBattleback.call(this);		
		    //this._fieldSprite = new Sprite(SceneManager.backgroundBitmap());
		    //this._fieldSprite.setBlendColor([16, 16, 16, 128]);
		    //this._fieldSprite.visible = false;
		};*/

		Spriteset_Battle.prototype.getActorSprite = function(index) //NEW
		{	
			return this._actorSprites[index];
		}

		Spriteset_Battle.prototype.getEnemySprite = function(index) //NEW
		{
			return this._enemySprites[this._enemySprites.length - 1 - index];
		}
	 	//END Spriteset_Battle
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Sprite_Battler
		Sprite_Battler.prototype._process_touch = function()
		{
			if(this._isMyTargeting())
			{
				if (TouchInput.isTriggered() && this._isTouched())  this._touching = true;
				if (this._touching) {
					if (TouchInput.isReleased() || !this._isTouched()) {
						this._touching = false;
						if (TouchInput.isReleased()) {
							 this._callClickHandler();
						}
					}
				}
			
			} 
			else 
			{
				this._touching = false;
			}
		}

		Sprite_Battler.prototype._isMyTargeting = function() {
			return false;
		}

		Sprite_Battler.prototype._isTouched = function() {
    		let x2 = this.screenX + this.screenW;
    		let y2 = this.screenY + this.screenH;
    		return ((TouchInput.x > this.screenX) && (TouchInput.x < x2) && (TouchInput.y > this.screenY) && (TouchInput.y < y2));
		}

		Sprite_Battler.prototype._callClickHandler = function() {
			//EMPTY
		}

		//END Sprite_Battler
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Sprite_ActorPKD
	 	class Sprite_ActorPKD extends Sprite_Actor
	 	{

	 		createMainSprite() {
			    this._mainSprite = new Sprite_Base();
			    this.addChild(this._mainSprite);
			    this._effectTarget = this._mainSprite;
			    this._touching = false;
			};

			setBattler(battler)
			{
				Sprite_Actor.prototype.setBattler.call(this, battler);
			}

	 		createShadowSprite() {
			    //EMPTY
			    this._shadowSprite = new Sprite();
			};

			createWeaponSprite() {
				//EMPTY
				this._weaponSprite = new Sprite_Weapon();
			};

			createStateSprite() {
				//EMPTY
				this._stateSprite = new Sprite_StateOverlay();
			};

			moveToStartPosition() {
			  	this.startMove(0, 300, 0);
			};

	 		updateBitmap()
	 		{
	 			Sprite_Battler.prototype.updateBitmap.call(this);
	 			var name = this._actor.battlerName();
			    if (this._battlerName !== name) {
			        this._battlerName = name;
			        this._mainSprite.bitmap = this._create_portrait();
			    }

			    this.updateDeadEffect();
	 		}

	 		updateDeadEffect() //NEW
	 		{
	 			if(!this._actor)
	 				return;

	 			var target = this._effectTarget;
	 			if(!this._actor.isAlive() && !this._battler.isSelected()) 
	 			{
	 				target.setBlendColor([255 , 128 , 128, 94]);
	 			}
	 			else {
	 				 target.setBlendColor([0, 0, 0, 0]);
	 			}
	 		}

	 		updateMotion()
	 		{
	 			//EMPTY
	 		}

	 		updateVisibility()
	 		{
	 			Sprite_Actor.prototype.updateVisibility.call(this);
			    if(BattleManagerRTBS.isBusy())
			    {
			    	this.visible = false;
			    	this._stopped_by_msg = true;
			    }
			   	else
			   	{
			   		if(this._stopped_by_msg == true)
			   		{
			   			this._stopped_by_msg = false;
			   			this.moveToStartPosition();
			   		}
			   	}
	 		}

	 		setActorHome(index) {
	 			if(index == 0)
	 			{
	 				this.setHome(4, 0);
	 			}
	 			else
	 			{
	 				this.setHome(2 + BattleSystem_Consts.PLAYER_WINDOW_W + ((index - 1) * (BattleSystem_Consts.ALLY_WINDOW_W + 2)), 0);
	 			}
			};

	 		updateFrame() {
			    //Sprite_Battler.prototype.updateFrame.call(this);	
			    if(!BattleManagerRTBS.isFastBattle() && BattleManagerRTBS.isBattle()) {
			    	this._process_touch();	
			    }
			};


	 		//PRIVATE
	 		_create_portrait() //NEW
	 		{
	 			let bitmap = new Bitmap(96,96);
	 			let faceIndex = this._actor.faceIndex();
	 			let faceName = this._actor.faceName();
				var pw = Window_Base._faceWidth;
		 		var ph = Window_Base._faceHeight;
		 		var sw = pw;
		 		var sh = ph;
	 			let sx = faceIndex % 4 * pw + (pw - sw) / 2;
	 			var sy = Math.floor(faceIndex / 4) * ph + (ph - sh) / 2;
	 			let temp = ImageManager.loadFace(faceName);
	 			temp.addLoadListener(function(){
	 				bitmap.blt(temp, sx, sy, sw, sh, 0, 0, 96, 96);
	 			}.bind(this));
	 			return bitmap;
	 		}

	 		_isMyTargeting()
	 		{
	 			var temp = BattleManagerRTBS.getPlayerInput();
	 			var target = false;
				if(temp) {
					target = !temp.targetInput().isTargetingForEnemy();
		 		}
		 		return target;
	 		}

	 		_isTouched() {
	 			let layer = BattleManagerRTBS.getLayers().playerLog;
	    		let x2 = this.screenX + this.screenW + layer.x;
	    		let y2 = this.screenY + this.screenH + layer.y;
	    		return ((TouchInput.x > this.screenX + layer.x) && (TouchInput.x < x2) && (TouchInput.y > this.screenY + layer.y) && (TouchInput.y < y2));
			}


		 	_callClickHandler() {
				let temp = BattleManagerRTBS.getPlayerInput();
				if(!temp) return;
				temp.targetInput().setExternalTarget(this._actor); //External target
				temp.onTargetClick(); //Extarnal action
			}
		}

	 	Object.defineProperties(Sprite_ActorPKD.prototype, {
			screenX : { get: function() {return this._homeX; }, configurable: true },
			screenY : { get: function() {return this._homeY; }, configurable: true },
			screenW : { get: function() {return this._mainSprite.width; }, configurable: true },
			screenH : { get: function() {return this._mainSprite.height; }, configurable: true },
			screenCX : { get: function() {return this.screenX + (this.screenW/2); }, configurable: true },
			screenCY : { get: function() {return this.screenY + (this.screenH/2); }, configurable: true }
		});

	 	//END Sprite_ActorPKD
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Sprite_Enemy
		function Sprite_Enemy_Consts() {
			throw new Error('This is a static class');
		}
		Sprite_Enemy_Consts.ANIM_STEP_PERCENT = 2; //Процент анимации движения (процент от картинки, на сколько сдвигать)
		Sprite_Enemy_Consts.ANIM_STEP_TIME = 32; //Время анимации движения (раз в ANIM_STEP_TIME кадров)

		let pkd_SpriteEnemy_setBattler = Sprite_Enemy.prototype.setBattler;
		Sprite_Enemy.prototype.setBattler = function(battler) {
		    pkd_SpriteEnemy_setBattler.call(this, battler);
		    this._animation_discale = Sprite_Enemy_Consts.ANIM_STEP_PERCENT / 100;
		    this._animation_flag = true;
		    this._animation_step = 1.0 / (Sprite_Enemy_Consts.ANIM_STEP_TIME * BattleSystem_Consts.FRAMES);
		    this._touching = false;
		    battler._sprite = this;
		};

		let pkd_SpriteEnemy_startEffect = Sprite_Enemy.prototype.startEffect;
		Sprite_Enemy.prototype.startEffect = function(effectType) {
			this._effectType = effectType;
			if(this._effectType == 'pkd_target') {
				this._effectDuration = 20;
				this.revertToNormal();
			} else {
				pkd_SpriteEnemy_startEffect.call(this, effectType);
			}
		}

		let pkd_SpriteEnemy_updateEffect = Sprite_Enemy.prototype.updateEffect;
		Sprite_Enemy.prototype.updateEffect = function() {
			if(this._effectType == 'pkd_target') {
				this.setupEffect();
				if (this._effectDuration > 0) {
       				 this._effectDuration--;
       				 this._update_target();

       				 if(this._effectDuration === 0) {
			            this._effectType = null;
			         }
       			}
			} else {
				pkd_SpriteEnemy_updateEffect.call(this);
			}
		}

		let pkd_SpriteEnemy_updatePosition = Sprite_Enemy.prototype.updatePosition;
		Sprite_Enemy.prototype.updatePosition = function() {
		    pkd_SpriteEnemy_updatePosition.call(this);
		    if(!BattleManagerRTBS.isFastBattle() && BattleManagerRTBS.isBattle()) {
		    	this._change_size();
		    	this._process_touch();
		    }
		};

		//PRIVATE
		Sprite_Enemy.prototype._update_target = function()//NEW, PKD Target selection effect 
		{
			var alpha = 128 - (16 - this._effectDuration) * 10;
    		this.setBlendColor([60, 210, 235, alpha]);
		}

		Sprite_Enemy.prototype._change_size = function() //NEW, Change size for animation
		{
			if(this._animation_flag === undefined) {
				return;
			}
			if(this._animation_flag) {
				this.scale.y -= this._animation_step;
			} else {
				this.scale.y += this._animation_step;
			}

			
			if((this.scale.y < (1.0 - this._animation_discale)) && this._animation_flag) {
				this._animation_flag = false;
				return;
			}

			if((this.scale.y > 1) && !this._animation_flag) {
				this._animation_flag = true;
			}
		}

		Sprite_Enemy.prototype._callClickHandler = function() {
			let temp = BattleManagerRTBS.getPlayerInput();
			if(!temp) return;
			temp.targetInput().setExternalTarget(this._enemy); //External target
			temp.onTargetClick(); //Extarnal action
		}

		Sprite_Enemy.prototype._isMyTargeting = function()
 		{
 			var temp = BattleManagerRTBS.getPlayerInput();
 			var target = false;
			if(temp) {
				target = temp.targetInput().isTargetingForEnemy();
	 		}
	 		return target;
 		}

		Object.defineProperties(Sprite_Enemy.prototype, {
			screenX : { get: function() {return (this._homeX - (this.bitmap.width/2)); }, configurable: true },
			screenY : { get: function() {return (this._homeY - this.height); }, configurable: true },
			screenW : { get: function() {return this.bitmap.width; }, configurable: true },
			screenH : { get: function() {return this.bitmap.height; }, configurable: true },
			screenCX : { get: function() {return this._homeX; }, configurable: true },
			screenCY : { get: function() {return this._homeY - (this.height/2); }, configurable: true }
		});
		//END Sprite_Enemy
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Sprite_Animation
		let pkd_SpriteAnimation_updatePosition = Sprite_Animation.prototype.updatePosition;
		Sprite_Animation.prototype.updatePosition = function() {
			var parent = this._target.parent;
			if(parent instanceof Sprite_ActorPKD)
			{
				if (this._animation.position === 3) { //Screen
			        this.x = this.parent.width / 2;
			        this.y = this.parent.height / 2;
			    } else {
	        		this.x = parent.x + this._target.height / 2;
	        		this.y = parent.y;

	        		if (this._animation.position === 0) { //Up
			            //this.y -= this._target.height;
			        } else if (this._animation.position === 1) { //Center
			            this.y += this._target.height / 2;
			        } else if (this._animation.position === 2) { //Down
			        	this.y += this._target.height;
			        }
			    }
			}
			else 
			{
				pkd_SpriteAnimation_updatePosition.call(this);
			}
		};

		let pkd_SpriteAnimation_updateCellSprite = Sprite_Animation.prototype.updateCellSprite;
		Sprite_Animation.prototype.updateCellSprite = function(sprite, cell) {
			pkd_SpriteAnimation_updateCellSprite.call(this, sprite, cell);
			if(this._target.parent instanceof Sprite_ActorPKD) {
				var pattern = cell[0];
				if (pattern >= 0) {
					sprite.x = 0;
        			sprite.y = 0;
					sprite.scale.x = (sprite.scale.x / 4);
					sprite.scale.y = (sprite.scale.y / 4);
				}
			}
		};

		//END Sprite_Animation
	//------------------------------------------------------------------------------

	//=============================================================================
	//SCENES
	//=============================================================================

	//------------------------------------------------------------------------------
		//Scene_Boot
		let pkd_SceneBoot_start = Scene_Boot.prototype.start;
		Scene_Boot.prototype.start = function() {
			pkd_SceneBoot_start.call(this);
			BattleManagerRTBS.init();
		}
		//END Scene_Boot
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Scene_Battle
		Scene_Battle.prototype.create = function() {
			Scene_Base.prototype.create.call(this);
			BattleManagerRTBS.initBattle();
	 		BattleManagerRTBS.setScene(this);
	 		this.createWindowLayer();
	 		this.createMessageWindow();
	 		this._process = new PKD_BattleSystem_SceneProcess();
		}

		Scene_Battle.prototype.update = function() {
			//if(Input.isTriggered(IKey.SPACE())) { //TEMP EXIT (Delete this if version is stable)
				//BattleManagerRTBS.abortBattle();
			//}
			this.updateBasic();
		}

		Scene_Battle.prototype.updateBasic = function() {
	 		let active = this.isActive();
			$gameTimer.update(active);
	 		$gameScreen.update();
			if(active && !this.isBusy())
 			{
 				this._process.update();
 			}
 			Scene_Base.prototype.update.call(this);
		}

		Scene_Battle.prototype.getPlayerInput = function() {
			return this._process.playerInput;
		}

		Scene_Battle.prototype.getBattleInput = function() {
			return this._process.battleInput;
		}

		Scene_Battle.prototype.start = function() {
		    Scene_Base.prototype.start.call(this);
		    this.startFadeIn(this.fadeSpeed(), false);
		    BattleManager.playBattleBgm();
		};

		Scene_Battle.prototype.battleStart = function() {
			this._process.battleStart();
		}

		Scene_Battle.prototype.onTurnEnd = function() {
			this._process.onTurnEnd();
		}

		Scene_Battle.prototype.hide = function() {
			this._process.hide();
		}

		Scene_Battle.prototype.stop = function() {
			Scene_Base.prototype.stop.call(this);
 			if(this.needsSlowFadeOut()) {
 				 this.startFadeOut(this.slowFadeSpeed(), false);
 			} else {
 				this.startFadeOut(this.fadeSpeed(), false);
 			} 
 			BattleManagerRTBS.stop();
		}
	 	//END Scene_Battle
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		// Scene_ItemBase
		let pkd_SceneItemBase_createActorWindow = Scene_ItemBase.prototype.createActorWindow;
		Scene_ItemBase.prototype.createActorWindow = function() {
			pkd_SceneItemBase_createActorWindow.call(this);
			this._circleWindow = new PKD_Window_SelectCircle();
			this._circleWindow.setOkHandler(this.onCircleItemOk.bind(this));
			this._circleWindow.setCancelHandler(this.onCircleItemCancel.bind(this));
			this.addWindow(this._circleWindow);
		}

		Scene_ItemBase.prototype.onCircleItemOk = function() { //NEW
			var item = this.item();
			if(!item)
				return;
			var index = this._circleWindow.getSelectedIndex();
			if(this._circleWindow.isSkillMode()) {
				this.user().addBattleSkill(item, index);
			}
			else {
				$gameParty.addBattleItem(item, index);
			}
			SoundManager.playUseItem();
			this._circleWindow.refresh();
		}

		Scene_ItemBase.prototype.onCircleItemCancel = function() { //NEW
			this.hideSubWindow(this._circleWindow);
		}
		// END Scene_ItemBase
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		// Scene_Item
		let pkd_SceneItem_onItemOk = Scene_Item.prototype.onItemOk;
		Scene_Item.prototype.onItemOk = function()
		{
			if(this._itemWindow.isBattleCategory()) {
				this._circleWindow.setItems();
				this.showSubWindow(this._circleWindow);
			} else {
				pkd_SceneItem_onItemOk.call(this);
			}
		}
		// END Scene_Item
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		// Scene_Skill
		let pkd_SceneSkill_onItemOk = Scene_Skill.prototype.onItemOk;
		Scene_Skill.prototype.onItemOk = function() {
			if(this._itemWindow.isBattleCategory()) {
				this._circleWindow.setSkills(this.user());
				this.showSubWindow(this._circleWindow);
			} else {
				pkd_SceneSkill_onItemOk.call(this);
			}
		}
		// END Scene_Skill
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Scene_Test
	    class PKD_Scene_Test extends Scene_Base
	    {
	    	constructor() {
	    		super();
	    	}

	    	create() {
	    		Scene_Base.prototype.create.call(this);
	    		this._draw_background();
	    		this.createWindowLayer();
	    		this.create_test();
	    	}

	    	update() {
	    		Scene_Base.prototype.update.call(this);
	    		this.update_test();
	    		//EXIT
				//if(Input.isTriggered('cancel') || TouchInput.isCancelled()) {
				if(Input.IsCancel()) {
					this.popScene();
				}
	    	}

	    	update_test() {
				//CUSTOM CODE
	    	}

	    	create_test() {
	    		//CUSTOM CODE
	    	}

	    	//RPIVATE
	    	_draw_background() {
	    		this._backgroundSprite = new Sprite();
		    	this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
		    	this._backgroundSprite.setBlendColor([16, 16, 16, 128]);
		    	this.addChild(this._backgroundSprite);
	    	}

	    }
		//END PKD_Scene_Test
	//------------------------------------------------------------------------------
	
	//=============================================================================
	//WINDOWS
	//=============================================================================
	
	//------------------------------------------------------------------------------
		//PKD_Window_SelectCircle
		class PKD_Window_SelectCircle extends Window_Base 
		{
			constructor()
			{
				super(0,0,(Graphics.boxWidth - 240),Graphics.boxHeight);
				this.circle = new PKD_Object_InputCircleSmall(260, 240, this, false);
				this.circle.showHelp();
				SDK.times(4, function(i) {
					this.circle.addListener(i, function() {
							this._process_click(i);
						}.bind(this));
				}.bind(this));
				this._mode = 'items'; //skills
				this._selectedIndex = -1;
				this._iconsArray = ['null','null','null','null'];
				this._helps = ['','','',''];
				this.hide();
			}

			setSkills(actor)
			{
				this._mode = 'skills';
				this._actor = actor;
				//this.refresh();
			}

			setItems()
			{
				this._mode = 'items';
				this._actor = null;
				//this.refresh();
			}

			isSkillMode()
			{
				return (this._mode == 'skills');
			}

			getSelectedIndex() {
				return this._selectedIndex;
			}

			setCancelHandler(func) {
				this._cancel = func;
			}

			setOkHandler(func) {
				this._ok = func;
			}

			refresh()
			{
				var items = [];
				if(this._mode == 'items') {
					items = $gameParty.getBattleItems();
				} else {
					items = this._actor.getBattleSkills();
				}

				for(let i = 0; i<items.length; i++) {
					if(items[i]) {
						this._iconsArray[i] = items[i].iconIndex;
						this._helps[i] = items[i].name;
					} else {
						this._iconsArray[i] = 'null';
						this._helps[i] = '';
					}
				}

				this.circle.setIcons(this._iconsArray);
				this.circle.setHelps(this._helps);
			}

			activate()
			{
				Window_Base.prototype.activate.call(this);
				this.refresh();
			}

			show()
			{
				Window_Base.prototype.show.call(this);
				this.circle.open();
			}

			hide()
			{
				Window_Base.prototype.hide.call(this);
				this.circle.close();
			}

			update()
			{
				Window_Base.prototype.update.call(this);
				this.circle.update();

				if(!this.active && !this.circle.isOpen()) return;

				if(Input.isTriggered(IKey.A())) {
					this._process_click(3);
				}

				if(Input.isTriggered(IKey.W())) {
					this._process_click(0);
				}

				if(Input.isTriggered(IKey.D())) {
					this._process_click(1);
				}

				if(Input.isTriggered(IKey.S())) {
					this._process_click(2);
				}

				//if(Input.isRepeated('cancel') || TouchInput.isCancelled()) {
				if(Input.IsCancel()) {
					SoundManager.playCancel();
					this._process_cancel();
				}

			}
			//PRIVATE
			_process_clickOk()
			{
				if(this._ok)
					this._ok();
			}

			_process_click(index) {
				this.circle.click(index);
				this._selectedIndex = index;
				this._process_clickOk();
			}

			_process_cancel()
			{
				this.deactivate();
				if(this._cancel)
					this._cancel();
			}
		}
		//END PKD_Window_SelectCircle
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Window_ItemCategory
		let pkd_WindowItemCategory_maxCols = Window_ItemCategory.prototype.maxCols;
		Window_ItemCategory.prototype.maxCols = function() {
		    return pkd_WindowItemCategory_maxCols.call(this) + 1;
		};

		let pkd_WindowItemCategory_makeCommandList = Window_ItemCategory.prototype.makeCommandList;
		Window_ItemCategory.prototype.makeCommandList = function() {
			this.addCommand(TextManager.fight,    'battle'); 
			pkd_WindowItemCategory_makeCommandList.call(this);
		}

		let pkd_WindowItemCategory_updateHelp = Window_ItemCategory.prototype.updateHelp;
		Window_ItemCategory.prototype.updateHelp = function() {
			pkd_WindowItemCategory_updateHelp.call(this);
			if (this._helpWindow) {
				if(this.index() == 0)
					this._helpWindow.setText(BattleSystem_Consts.STRING_ITEMS_HELP[SDK.isRU()]);
			} 
		}
		//END Window_ItemCategory
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Window_ItemList
		let pkd_WindowItemList_includes = Window_ItemList.prototype.includes;
		Window_ItemList.prototype.includes = function(item) {
		    if(this.isBattleCategory()) {
		    	return DataManager.isItem(item) && item.itypeId === 1 && (item.occasion === 0 || item.occasion === 1);
		    } else {
		    	return pkd_WindowItemList_includes.call(this, item);
		    }
		};

		let pkd_WindowItemList_isEnabled = Window_ItemList.prototype.isEnabled;
		Window_ItemList.prototype.isEnabled = function(item) {
			if(this.isBattleCategory()) {
				return true; 
			} else return pkd_WindowItemList_isEnabled.call(this, item);
		};

		Window_ItemList.prototype.isBattleCategory = function() //NEW
		{
			return (this._category == 'battle');
		}
		//END Window_ItemList
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Window_SkillType
		let pkd_WindowSkillType_makeCommandList = Window_SkillType.prototype.makeCommandList;
		Window_SkillType.prototype.makeCommandList = function() {
			if (this._actor) {
				this.addCommand(TextManager.fight, 'skill', true, 'battle'); 
			}
			pkd_WindowSkillType_makeCommandList.call(this);
		}

		let pkd_WindowSkillType_updateHelp = Window_SkillType.prototype.updateHelp;
		Window_SkillType.prototype.updateHelp = function() {
			pkd_WindowSkillType_updateHelp.call(this);
			if (this._helpWindow) {
				if(this.index() == 0)
					this._helpWindow.setText(BattleSystem_Consts.STRING_ITEMS_HELP[SDK.isRU()]);
			} 
		}
		//END Window_SkillType
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Window_SkillList
		let pkd_Window_SkillList_includes = Window_SkillList.prototype.includes;
		Window_SkillList.prototype.includes = function(item) {	
			if(this.isBattleCategory()) {
				return DataManager.isSkill(item) && (item.occasion === 0 || item.occasion === 1);
			} else return pkd_Window_SkillList_includes.call(this, item); 
		}

		let pkd_Window_SkillList_isEnabled = Window_SkillList.prototype.isEnabled;
		Window_SkillList.prototype.isEnabled = function(item) {
			if(this.isBattleCategory()) {
				return true;
			} else return pkd_Window_SkillList_isEnabled.call(this, item);
		}

		Window_SkillList.prototype.isBattleCategory = function() //NEW
		{
			return (this._stypeId == 'battle');
		}
		//END Window_SkillList
	//------------------------------------------------------------------------------

	//=============================================================================
	//OBJECTS
	//=============================================================================

	//------------------------------------------------------------------------------
	 	//PKD_Object_SCircle
	 	function PKD_Object_SCircle_Consts() {
	 		throw new Error('This is a static class');
	 	}

 		//Время анимции клика (в кадрах)
		PKD_Object_SCircle_Consts.CLICK_TIME = 5;  
		//Цвет при клике на сегмент
		PKD_Object_SCircle_Consts.COLOR_CLICK = new ColorPKD(98,225,236,220);
		//Цвет выбраного сегмента
		PKD_Object_SCircle_Consts.COLOR_SELECTED = new ColorPKD(164,255,164,220);
		//Цвет отключённого сегмента
		PKD_Object_SCircle_Consts.COLOR_DISABLED = new ColorPKD(89,89,89,120);

		class PKD_Object_SCircle
	    {
	    	constructor(x, y, parent, isOpen)
	    	{
	    		if(isOpen === undefined)
	    			isOpen = true;

	    		this.x = x;
	    		this.y = y;
	    		this._isOpen = isOpen;
	    		this.isClicked = false;
	    		this.isSelected = false;
	    		this.icons = [];
	    		this.segments = [];
	    		this.helps = [];
	    		this._helpsSprites = [];
	    		this.optionLightTimer = 0;
	    		this.parent = parent;

	    		//this.helpAllwaysShow = false;
	    		this.helpShowIn = false;
	    		this._helpIsHidden = true;

	    		let radius = this._radius_max()
	    		if(!this._isOpen) {
	    			this._set_opacity(0)
	    			radius = this._radius_min()
	    		}

				this._create_segments();
				this._create_icons(this._get_standart_icon_pack());
	    		this._move_segments(x,y,radius);

	    		if(this._isOpen == false)
	    			this._set_opacity(0);
	    	}

	    	isOpen()
	    	{
	    		return this._isOpen;
	    	}

	    	open() 
	    	{
	    		if(this._isOpen) return;
	    		if(this.isAnimate) return;
	    		this.r = this._radius_min();
	    		this._set_opacity(0);
	    		this.isAnimate = true;
	    	}

	    	close()
	    	{
	    		if(!this._isOpen) return;
	    		if(this.isAnimate) return;
	    		this.r = this._radius_max();
	    		this._set_opacity(255);
	    		this.isAnimate = true;
	    		this._hide_help();
	    	}

	    	showHelp()
	    	{
	    		this.helpShowIn = true;
	    		//this.helpAllwaysShow = true;
	    	}

	    	hideHelp()
	    	{
	    		this.helpShowIn = false;
	    		//this.helpAllwaysShow = false;
	    	}

	    	click(index)
	    	{
	    		this.diselectAllIcons();
	    		this.optionLightTimer = 0;
	    		this.isClicked = true;
	    		this.segments[index].setBlendColor(PKD_Object_SCircle_Consts.COLOR_CLICK.toArr());
	    	}

	    	select(index)
	    	{
	    		this.diselectAllIcons();
	    		this.segments[index].setBlendColor(PKD_Object_SCircle_Consts.COLOR_SELECTED.toArr());
	    		this.isSelected = true;
	    	}

	    	hideSegment(index, isHided)
	    	{
	    		if(isHided === undefined)
	    			isHided = true;
	    		this.segments[index].visible = !isHided;
	    		if(this.icons[index])
	    			this.icons[index].visible = !isHided;

	    		if(this._helpsSprites[index])
	    		{
	    			if(isHided == true) {
	    				this._helpsSprites[index].visible = false;
	    			} else {
	    				if(this.helpAllwaysShow || this.helpShowIn)
	    					this._helpsSprites[index].visible = true;
	    			}
	    		}
	    	}

	    	disableIcon(index, isDisabled)
	    	{
	    		if(isDisabled === undefined)
	    			isDisabled = true;
	    		if(!this.icons[index])
	    			return;
	    		else
	    		{
	    			if(isDisabled) {
	    				this.icons[index].setBlendColor(PKD_Object_SCircle_Consts.COLOR_DISABLED.toArr());
	    			}
	    			else {
	    				this.icons[index].setBlendColor(ColorPKD.NONE_Arr);
	    			}
	    		}
	    	}

	    	setIcons(arrayOfIndexes)
	    	{
	    		this._create_icons(arrayOfIndexes);
	    		this._move_icons();
	    		if(!this._isOpen)
	    			this._set_opacity(0);
	    	}

	    	setHelps(arrayOfHelps)
	    	{
	    		this.helps[0] = arrayOfHelps[0] || null;
	    		this.helps[1] = arrayOfHelps[1] || null;
	    		this.helps[2] = arrayOfHelps[2] || null;
	    		this.helps[3] = arrayOfHelps[3] || null;
	    		if(this.helpShowIn) {
	    			this._helpIsHidden = true;
	    		}
	    		this._create_helps();
	    	}

	    	hideAll()
	    	{
	    		SDK.times(4, function(i) { this.hideSegment(i); }.bind(this));
	    	}

	    	enableAllIcons()
	    	{
	    		SDK.times(4, function(i) { this.disableIcon(i, false);}.bind(this));
	    	}

	    	diselectAllIcons()
	    	{
	    		this._reset_colors();
	    		this.isSelected = false;
	    	}

	    	unHideSegments()
	    	{
	    		SDK.times(4, function(i) { this.hideSegment(i, false); }.bind(this));
	    	}

	    	reset()
	    	{
	    		this.unHideSegments();
	    		this.diselectAllIcons();
	    		this.enableAllIcons();
	    	}

	    	addListener(index, func)
	    	{
	    		this.segments[index].setClickHandler(func);
	    	}

	    	update()
	    	{
	    		this.segments.forEach(function(x) {
	    			x.update();
	    		});
	    		this._update_animation();
	    		this._update_click();
	    		this._update_help();
	    	}

	    	//PRIVATE
	    	_create_segments()
	    	{
	    		let csT = new Sprite_Button();
	    		csT.bitmap = ImageManager.loadPKD('CircleSegment');
	    		let csD = new Sprite_Button();
	    		csD.bitmap = ImageManager.loadPKD('CircleSegment_down');
	    		let csL = new Sprite_Button();
	    		csL.bitmap = ImageManager.loadPKD('CircleSegment_L');
	    		let csR = new Sprite_Button();
	    		csR.bitmap = ImageManager.loadPKD('CircleSegment_R');
				this.segments = [csT, csR, csD, csL];
				let sceneX = this.parent
				this.segments.forEach(function(x) { sceneX.addChild(x); });
	    	}

	    	_create_icons(arrayOfIndexes)
	    	{
	    		this.icons.forEach(function(x){ 
	    			this.parent.removeChild(x);
	    		}.bind(this));
	    		this.icons = [];

	    		for(let i = 0; i<4; i++) {
	    			if(arrayOfIndexes[i] !== undefined)
	    			{
	    				this.icons[i] = new Sprite(this._get_icon_bitmap(arrayOfIndexes[i]));
	    				this.parent.addChild(this.icons[i]);
	    			}
	    			else {
	    				this.icons[i] = undefined;
	    			}
	    		}
	    	}

	    	_create_helps()
	    	{
	    		this._helpsSprites.forEach(function(item){
	    			if(item!=null) {
	    				this.parent.removeChild(item);
	    			}
	    		}.bind(this));
	    		this._helpsSprites = [];

	    		//UP
	    		if(this.helps[0] != null) {
		    		let bitmap = new Bitmap(this._segment_HW(), this._segment_HH());
		    		bitmap.addLoadListener(function() {
		    			bitmap.drawText(this.helps[0], 0, 0, bitmap.width, bitmap.height, 'center');
		    		}.bind(this));
		    		let sprite = new Sprite(bitmap);
		    		sprite.x = this.segments[0].x;
		    		sprite.y = this.segments[0].y - this._segment_HH();
		    		sprite.visible = false;
		    		this.parent.addChild(sprite);
		    		this._helpsSprites[0] = sprite;
	    		}
	    		else {
	    			this._helpsSprites[0] = null;
	    		}
	    		//LEFT
	    		if(this.helps[3] != null) {
		    		let bitmap = new Bitmap(this._segment_HW(), this._segment_HH());
		    		bitmap.addLoadListener(function() {
		    			bitmap.drawText(this.helps[3], 0, 0, bitmap.width, bitmap.height, 'center');
		    		}.bind(this));
		    		let sprite = new Sprite(bitmap);
		    		sprite.x = this.segments[0].x - (this._segment_HW() * 1.4);
		    		sprite.y = this.segments[0].y + (this._segment_VH() / 1.6);
		    		sprite.visible = false;
		    		this.parent.addChild(sprite);
		    		this._helpsSprites[3] = sprite;
	    		}
	    		else {
	    			this._helpsSprites[3] = null;
	    		}
	    		//DOWN
	    		if(this.helps[2] != null) {
		    		let bitmap = new Bitmap(this._segment_HW(), this._segment_HH());
		    		bitmap.addLoadListener(function() {
		    			bitmap.drawText(this.helps[2], 0, 0, bitmap.width, bitmap.height, 'center');
		    		}.bind(this));
		    		let sprite = new Sprite(bitmap);
		    		sprite.x = this.segments[2].x;
		    		sprite.y = this.segments[2].y + (this._segment_HH() * 1.4);
		    		sprite.visible = false;
		    		this.parent.addChild(sprite);
		    		this._helpsSprites[2] = sprite;
	    		}
	    		else {
	    			this._helpsSprites[2] = null;
	    		}
	    		//RIGHT
	    		if(this.helps[1] != null) {
		    		let bitmap = new Bitmap(this._segment_HW(), this._segment_HH());
		    		bitmap.addLoadListener(function() {
		    			bitmap.drawText(this.helps[1], 0, 0, bitmap.width, bitmap.height, 'center');
		    		}.bind(this));
		    		let sprite = new Sprite(bitmap);
		    		sprite.x = this.segments[0].x + (this._segment_HW() * 1.4);
		    		sprite.y = this.segments[0].y + (this._segment_VH() / 1.6);
		    		sprite.visible = false;
		    		this.parent.addChild(sprite);
		    		this._helpsSprites[1] = sprite;
	    		}
	    		else {
	    			this._helpsSprites[1] = null;
	    		}

	    	}

	    	_get_standart_icon_pack()
	    	{
	    		return [IKey.W(),IKey.D(),IKey.S(),IKey.A()];
	    	}

	    	_get_icon_bitmap(index)
	    	{
	    		return ImageManager.getButtonPKD(index,this._icon_size());
	    	}

	    	_move_segments(x, y, radius)
	    	{
	    		if(radius === undefined) {
	    			radius = this._radius_max();
	    		}

	    		//TOP
				this.segments[0].x = x 
				this.segments[0].y = y
				
				//DOWN
				this.segments[2].x = this.segments[0].x
				this.segments[2].y = this.segments[0].y + radius
				
				//LEFT
				this.segments[3].x = x - Math.floor(radius / 4)
				this.segments[3].y = Math.floor(this.segments[0].y + (radius + this._segment_HH() - 
					this._segment_VH())/2)
				
				//RIGHT
				this.segments[1].x = x + Math.floor(radius / 2) + 
							Math.floor(this._segment_VW() / 2 + this._segment_VW() / 4) - 3
				this.segments[1].y = this.segments[3].y

				this._move_icons()
	    	}

	    	_move_icons()
	    	{
	    		if(this.icons[0])
	    		{
	    			let iT = this.icons[0];
	    			iT.x = this.segments[0].x + this._segment_HW()/2 - this._icon_size()/2;
	    			iT.y = this.segments[0].y + 3;
	    		}

	    		if(this.icons[1])
	    		{
	    			let iR = this.icons[1];
	    			iR.x = this.segments[1].x - 5 + this._icon_size()/2;
	    			iR.y = this.segments[3].y + this._segment_VH()/2 - this._icon_size()/2;
	    		}

	    		if(this.icons[2])
	    		{
	    			let iD = this.icons[2];
	    			iD.x = this.segments[2].x + this._segment_HW()/2 - this._icon_size()/2;
	    			iD.y = this.segments[2].y + 13;
	    		}

	    		if(this.icons[3])
	    		{
	    			let iL = this.icons[3];
	    			iL.x = this.segments[3].x + 3
	    			iL.y = this.segments[3].y + this._segment_VH()/2 - this._icon_size()/2;
	    		}

	    	}

	    	_update_animation()
	    	{
	    		if(!this.isAnimate) return;
	    		if(this.isClicked) return;

	    		this._move_segments(this.x,this.y,this.r);
	    		if(!this._isOpen)
	    		{
	    			if(this._opacity() <= 245) this._change_opacity(10);
	    			if (this.r < this._radius_max()) this.r += 2;
	    			if (this.r == this._radius_max()) {
	    				this._isOpen = true;
	    				this.isAnimate = false;
	    				this._set_opacity(255);
	    			}
	    		}
	    		else 
	    		{
	    			if(this._opacity() > 10) this._change_opacity(10, false);
	    			if (this.r > this._radius_min()) this.r -= 2;
	    			if (this.r == this._radius_min()) {
	    				this._isOpen = false;
	    				this.isAnimate = false;
	    				this._set_opacity(0);
	    			}
	    		}
	    	}

	    	_update_click()
	    	{
	    		if (this.optionLightTimer < PKD_Object_SCircle_Consts.CLICK_TIME)
	    			this.optionLightTimer += 1;

	    		if(this.optionLightTimer == PKD_Object_SCircle_Consts.CLICK_TIME)
	    		{
	    			if(!this.isSelected)
	    				this._reset_colors();
	    			this.isClicked = false;
	    		}
	    	}

	    	_update_help()
	    	{
	    		if(this.isAnimate)
	    		{
	    			this._hide_help();
	    		}
	    		else {
	    			if(this._isOpen)
			    		if(this.helpShowIn)
			    			this._show_help();
			    		else
			    			this._hide_help(); 
			    	else
			    		this._hide_help();
	    		}
	    	}

	    	_show_help()
	    	{
	    		if(this._helpIsHidden == false)
	    			return;

	    		for(var i = 0; i<4; i++) {
	    			let sprite = this._helpsSprites[i] || null;
	    			if(sprite !=null) {
	    				sprite.visible = (this.segments[i].visible == true);
	    			}
	    		}

	    		this._helpIsHidden = false;
	    	}

	    	_hide_help()
	    	{
	    		if(this._helpIsHidden == true)
	    			return;
				for(var i = 0; i<4; i++) {
	    			let sprite = this._helpsSprites[i] || null;
	    			if(sprite !=null) {
	    				sprite.visible = false;
	    			}
	    		}
	    		this._helpIsHidden = true;
	    	}

	    	_change_opacity(value, isAdd)
	    	{
	    		if(isAdd === undefined)
	    			isAdd = true;

	    		let op = this._opacity();
	    		if (isAdd) { op += value; }
	    			else { op -= value; }
	    		this._set_opacity(op);
	    	}

	    	_set_opacity(value)
	    	{
	    		this.segments.forEach(function(x) {
	    			x.opacity = value;
	    		});
	    		this.icons.forEach(function(x) { 
	    			if(x) 
	    				x.opacity = value; 
	    		});
	    	}

	    	_opacity()
	    	{
	    		if(this.segments[0]) { return this.segments[0].opacity; }
	    		return 0;
	    	}

	    	_reset_colors()
	    	{
	    		this.segments.forEach(function(x) { x.setBlendColor(ColorPKD.NONE_Arr); });
	    	}

	    	//VALUES

	    	_segment_HW()
	    	{
	    		return 132;
	    	}

	    	_segment_HH()
	    	{
	    		return 54;
	    	}

	    	_segment_VW()
	    	{
	    		return this._segment_HH();
	    	}

	    	_segment_VH()
	    	{
	    		return this._segment_HW();
	    	}

	    	_radius_max()
	    	{
	    		return 165;
	    	}

	    	_radius_min()
	    	{
	    		return 145;
	    	}

	    	_icon_size()
	    	{
	    		return 38;
	    	}

	    }
		//END PKD_Object_SCircle
	//------------------------------------------------------------------------------
	 
	//------------------------------------------------------------------------------
	 	//PKD_Object_SCircleSmall
	 	class PKD_Object_SCircleSmall extends PKD_Object_SCircle
	 	{
	 		//PRIVATE
	 		_create_segments()
	 		{
	    		let csT = new Sprite_Button();
	    		csT.bitmap = ImageManager.loadPKD('CircleSegment_small');
	    		let csD = new Sprite_Button();
	    		csD.bitmap = ImageManager.loadPKD('CircleSegment_small_down');
	    		let csL = new Sprite_Button();
	    		csL.bitmap = ImageManager.loadPKD('CircleSegment_small_L');
	    		let csR = new Sprite_Button();
	    		csR.bitmap = ImageManager.loadPKD('CircleSegment_small_R');
				this.segments = [csT, csR, csD, csL];
				let sceneX = this.parent
				this.segments.forEach(function(x) { sceneX.addChild(x) });
	 		}

	 		_create_helps()
	 		{
	 			PKD_Object_SCircle.prototype._create_helps.call(this);
	 			//if(this._helpsSprites[2] != null)
	 			//	this._helpsSprites[2].y -= 14;
	 		}

	 		_move_icons()
	 		{
	 			//TOP
	 			if(this.icons[0])
	    		{
	    			let iT = this.icons[0];
	    			iT.anchor.x = 0.4
	    			iT.x = this.segments[0].x + this._segment_HW() / 2;
	    			iT.y = this.segments[0].y;
	    		}

	    		//RIGHT
	    		if(this.icons[1])
	    		{
	    			let iR = this.icons[1];
	    			iR.x = this.segments[1].x + this._icon_size()/2 - 3;
	    			iR.y = this.segments[3].y + this._segment_VH()/2 - this._icon_size()/2;
	    		}

	    		//DOWN
	    		if(this.icons[2])
	    		{
	    			let iD = this.icons[2];
	    			iD.anchor.x = 0.4
	    			iD.x = this.segments[2].x + this._segment_HW() / 2;
	    			iD.y = this.segments[2].y + 9;
	    		}

	   			//LEFT
	    		if(this.icons[3])
	    		{
	    			let iL = this.icons[3];
	    			iL.x = this.segments[3].x + 1
	    			iL.y = this.segments[3].y + this._segment_VH()/2 - this._icon_size()/2;
	    		}
	 		}

	    	_get_standart_icon_pack()
	    	{
	    		return ['items','defense','attack','spells'];
	    	}

	 		_get_icon_bitmap(index)
	 		{
	 			return ImageManager.getIconPKD(index, this._icon_size());
	 		}

	 		//VALUES
	 		_segment_HW()
	    	{
	    		return 77;
	    	}

	    	_segment_HH()
	    	{
	    		return 34;
	    	}

	    	_radius_max()
	    	{
	    		return 95;
	    	}

	    	_radius_min()
	    	{
	    		return 75;
	    	}

	    	_icon_size()
	    	{
	    		return 24;
	    	}
	 	}
	 	//END PKD_Object_SCircleSmall
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_InputCircle
	 	class PKD_Object_InputCircle
	 	{
	 		constructor(x, y, parent, helpShow)
	 		{
	 			this.x = x;
	 			this.y = y;
	 			this.parent = parent;
	 			this.circle = new PKD_Object_SCircle(x, y, parent, false);
	 			let xs = (x + this.circle._segment_HW() / 4) - 7;
	 			let ys = (y + this.circle._segment_VW()) - 7;
	 			this.ySmall = ys;
				this.circle2 = new PKD_Object_SCircleSmall(xs, ys, parent, false);

				helpShow = SDK.check(helpShow, false);
				if(helpShow)
					this.circle.helpAllwaysShow = true;
	 		}

	 		isOpen()
	 		{
	 			return this.circle.isOpen();
	 		}

	 		open()
	 		{
	 			this._perform('open');
	 		}

	 		close()
	 		{
	 			this._perform('close');
	 		}

	 		showHelp()
	 		{
	 			this.circle.showHelp();
	 		}

	 		hideHelp()
	 		{
	 			this.circle.hideHelp();
	 		}

	 		setHelps(arrayOfHelps)
	 		{
	 			this.circle.setHelps(arrayOfHelps);
	 		}

	 		click(index)
	 		{
	 			SoundManager.playOk();
	 			this._perform('click', [index]);
	 		}

	 		select(index)
	 		{
	 			SoundManager.playCursor();
	 			this._perform('select', [index]);
	 		}

	 		disableIcon(index, isDisabled)
	 		{
	 			if(isDisabled === undefined)
	 				isDisabled = true;
	 			this._perform('disableIcon', [index, isDisabled]);
	 		}

	 		hideSegment(index, isHided)
	 		{
	 			if(isHided === undefined)
	 				isHided = true;
	 			this._perform('hideSegment',[index,isHided]);
	 		}

	 		enableAllIcons()
	 		{
	 			this._perform('enableAllIcons');
	 		}

	 		diselectAllIcons()
	 		{
	 			this._perform('diselectAllIcons');
	 		}

	 		unHideSegments()
	 		{
	 			this._perform('unHideSegments');
	 		}

	 		reset()
	 		{
	 			this._perform('reset');
	 		}

	 		update()
	 		{
	 			this._perform('update');
	 		}

	 		addListener(index, func)
	 		{
	 			this.circle.addListener(index, func);
	 		}

	 		//PRIVATE
	 		_perform(funcName, args)
	 		{
	 			this.circle[funcName].apply(this.circle, args);
	 			this.circle2[funcName].apply(this.circle2, args);
	 		}

	 	}
	 	//END PKD_Object_InputCircle
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_CircleInnerButtons
	 	function PKD_Object_CircleInnerButtons_Consts() {
	 		throw new Error('This is a static class');
	 	}
	 	PKD_Object_CircleInnerButtons_Consts.COLOR_SELECTED = new ColorPKD(164,255,164,100);

	 	class PKD_Object_CircleInnerButtons
	 	{
	 		constructor(x, y, parent, isOpen)
	 		{
	 			if(isOpen === undefined)
	 				isOpen = true;

	 			this._isOpen = isOpen;
	 			this.x = x;
	 			this.y = y;
	 			this.parent = parent;
	 			this.buttons = [];
	 			this.disabledArray = [];

	 			this._create_buttons();

	 			if(!this._isOpen) {
	 				this._set_visible_all(false);
	 			}
	 		}

	 		isOpen()
	 		{
	 			return this._isOpen;
	 		}

	 		open()
	 		{
	 			if(this._isOpen) return;
	 			this._set_visible_all(true);
	 			this._isOpen = true;
	 		}

	 		close()
	 		{
	 			if(!this._isOpen) return;
				this._set_visible_all(false);
				this._isOpen = false;

	 		}

	 		select(index)
	 		{
	 			this.diselectAllButtons();
	 			this._select_on_keyboard(index);
	 		}

	 		disableButton(index, isDisabled)
	 		{
	 			if(isDisabled === undefined)
	 				isDisabled = true;
	 			if(isDisabled) {
	 				if(!this.disabledArray.include(index)) {
	 					this.disabledArray.push(index);
	 				}
	 				this.buttons[index].setBlendColor(PKD_Object_SCircle_Consts.COLOR_DISABLED.toArr());
	 			}
	 			else {
	 				this.disabledArray.delete(index);
	 				this.buttons[index].setBlendColor(ColorPKD.NONE_Arr);
	 			}
	 		}

	 		hideButton(index, isHided)
	 		{
	 			isHided = SDK.check(isHided, true);
	 			if(isHided)
	 				this.buttons[index].opacity = 0;
				else
	 				this.buttons[index].opacity = 200;
	 		}

	 		enableAllButtons()
	 		{
	 			SDK.times(4, function(i) {
	 				this.disableButton(i, false);
	 			}.bind(this));
	 		}

	 		unHideButtons()
	 		{
	 			SDK.times(4, function(i) {
	 				this.hideButton(i, false);
	 			}.bind(this));
	 		}

	 		diselectAllButtons()
	 		{
	 			this._reset_colors();
	 		}

	 		reset()
	 		{
	 			this.unHideButtons();
	 			this.diselectAllButtons();
	 			this.enableAllButtons();
	 		}

	 		update()
	 		{
	 			this.buttons.forEach(function(x){ if(x) x.update(); });
	 		}


	 		//PRIVATE

	 		_create_buttons()
	 		{
	 			//UP
	 			this.buttons[0] = new Sprite(ImageManager.getButtonPKD(IKey.W(), 24));
	 			//RIGHT
	 			this.buttons[1] = new Sprite(ImageManager.getButtonPKD(IKey.D(), 24));
	 			//DOWN
	 			this.buttons[2] = new Sprite(ImageManager.getButtonPKD(IKey.S(), 24));
	 			//LEFT
	 			this.buttons[3] = new Sprite(ImageManager.getButtonPKD(IKey.A(), 24));

	 			SDK.times(4, function(i) { this.parent.addChild(this.buttons[i]);}.bind(this));

	 			this.buttons[0].opacity = 200;
	 			this.buttons[0].x = this.x + 27;
	 			this.buttons[0].y = this.y + 28;

	 			this.buttons[1].opacity = this.buttons[0].opacity;
	 			this.buttons[1].x = this.x + 49;
	 			this.buttons[1].y = this.y + 52;

	 			this.buttons[2].opacity = this.buttons[0].opacity;
	 			this.buttons[2].x = this.x + 27;
	 			this.buttons[2].y = this.y + 76;

	 			this.buttons[3].opacity = this.buttons[0].opacity;
	 			this.buttons[3].x = this.x + 4;
	 			this.buttons[3].y = this.y + 52;
	 		}

	 		_reset_colors()
	 		{
	 			this.buttons.forEach(function(x) { x.setBlendColor(ColorPKD.NONE_Arr); });
	 			this.disabledArray.forEach(function(i) { 
	 				this.buttons[i].setBlendColor(PKD_Object_SCircle_Consts.COLOR_DISABLED.toArr()); 
	 			}.bind(this));
	 		}

	 		_select_on_keyboard(index)
	 		{
	 			if(this.disabledArray.include(index)) return;
	 			this.buttons[index].setBlendColor(PKD_Object_CircleInnerButtons_Consts.COLOR_SELECTED.toArr());
	 		}

	 		_set_visible_all(isVisible)
	 		{
	 			if(isVisible === undefined)
	 				isVisible = true;
	 			this.buttons.forEach(function(x) { this._set_visible(x, isVisible);  }.bind(this));
	 		}

	 		_set_visible(sprite, isVisible)
	 		{
	 			if(isVisible === undefined)
	 				isVisible = true;
	 			if(sprite)
	 				sprite.visible = isVisible;
	 		}

	 	}
	 	//END PKD_Object_CircleInnerButtons
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_InputCircleSmall
	 	class PKD_Object_InputCircleSmall extends PKD_Object_SCircleSmall
	 	{
	 		constructor(x,y,parent,isOpen)
	 		{
	 			super(x,y,parent,isOpen);
	 			this.buttons = new PKD_Object_CircleInnerButtons(x,y,parent,isOpen);
	 		}

	 		close()
	 		{
	 			PKD_Object_SCircleSmall.prototype.close.call(this);
	 			this.buttons.close();
	 		}

	 		//setIcons(arrayOfIndexes)
	 		//{
	 		//	
	 		//}

	 		select(index)
	 		{
	 			SoundManager.playCursor();
	 			PKD_Object_SCircleSmall.prototype.select.call(this, index);
	 			this.buttons.select(index);
	 		}

	 		disableIcon(index, isDisabled)
	 		{
	 			PKD_Object_SCircleSmall.prototype.disableIcon.call(this, index, isDisabled);
	 			this.buttons.disableButton(index, isDisabled);
	 		}

	 		hideSegment(index, isHided)
	 		{
	 			PKD_Object_SCircleSmall.prototype.hideSegment.call(this, index, isHided);
	 			this.buttons.hideButton(index, isHided);
	 		}

	 		enableAllIcons()
	 		{
				PKD_Object_SCircleSmall.prototype.enableAllIcons.call(this);
				this.buttons.enableAllButtons();
	 		}

	 		diselectAllIcons()
	 		{
	 			PKD_Object_SCircleSmall.prototype.diselectAllIcons.call(this);
	 			this.buttons.diselectAllButtons();
	 		}

	 		unHideSegments()
	 		{
	 			PKD_Object_SCircleSmall.prototype.unHideSegments.call(this);
	 			this.buttons.unHideButtons();
	 		}

	 		update()
	 		{
	 			PKD_Object_SCircleSmall.prototype.update.call(this);
	 			this.buttons.update();
	 			this._update_buttons();
	 		}

	 		//PRIVATE
	 		_update_buttons()
	 		{
	 			if(this._isOpen) {
	 				if(!this.isAnimate) {
	 					if(this._opacity() >= 255) {
	 						this.buttons.open();
	 					}
	 				}
	 			}
	 		}
	 	}
	 	//END PKD_Object_InputCircleSmall
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_Object_EnemyInfo
		class PKD_Object_EnemyInfo
		{
			constructor(battler, layer)
			{
				this.w = 100; //width
				this._battler = battler;
				this._layer = layer;
				this._selected = false;
				this._iconIndex = null; //This icon will be showed on enemy head
				this._create_sprites();
			}

			select(isSelected, iconIndex)
			{
				if(this._battler.isAlive()) {
					this._iconIndex = iconIndex;
					this._selected = isSelected;
					if(isSelected) {
						this._battler.sprite.startEffect('pkd_target');
					}
				} else {
					this._iconIndex = null;
					this._selected = false;
				}
				this._draw_icon();
			}

			update()
			{
				if(this._battler.isAlive()) {
					this._sprite_info.visible = true;
					this._update_info();
				} else {
					this._selected = false;
					this._sprite_info.visible = false;
				}
			}

			hide() {
				this._sprite_info.visible = false;
			}

			//PRIVATE
			_update_info()
			{
				if(this._selected){
					this._sprite_info.opacity = 255;
				} else {
					this._sprite_info.opacity = 120;
				}
				this._bar_hp.update();
			}

			_create_sprites()
			{
				let w = this.w;
				let wx = w / 2;
				let bitmap = new Bitmap(w, 46);
				bitmap.addLoadListener(function() {
					//bitmap.fillRect(0,0,bitmap.width, bitmap.height, ColorPKD.GREEN());
					this._bar_hp = new PKD_Object_BarHPE(bitmap, this._battler);
					this._bar_hp.setPosition(wx-(wx/2),26,wx,16);
				}.bind(this));

				this._sprite_info = new Sprite(bitmap);
				//this._battler.sprite.bitmap.fillRect(0,0,this._battler.sprite.bitmap.width, this._battler.sprite.bitmap.height, ColorPKD.RED());
				let temp = this._battler.sprite.screenW - w;
				this._sprite_info.x = this._battler.sprite.screenX + (temp/2);
				this._sprite_info.y = this._battler.sprite.screenY - 28;

				this._sprite_icon = new Sprite(this._get_icon());
				this._sprite_icon.x = this._sprite_info.x + wx - 12;
				this._sprite_icon.y = this._sprite_info.y - 26;

				this._layer.addChild(this._sprite_info);
				this._layer.addChild(this._sprite_icon);

				this._refresh();
			}

			_draw_icon()
			{
				if(this._iconIndex) {
					this._sprite_icon.bitmap = this._get_icon();
				} else {
					this._sprite_icon.bitmap = ImageManager.getIconPKD('null', 24);
				}
			}

			_get_icon()
			{
				return ImageManager.getIconPKD(this._iconIndex, 24, true);
			}

			_refresh()
			{
				this._sprite_info.bitmap.clear();
				//this._sprite_info.bitmap.fillRect(0,0,this._sprite_info.bitmap.width, this._sprite_info.bitmap.height, ColorPKD.GREEN());
				//this._sprite_info.bitmap.fontFace = BattleSystem_Consts.FONT;
				this._sprite_info.bitmap.drawText(this._battler.name(), 0, 0, this.w, 24, 'center');
			}
		}
		//END PKD_Object_EnemyInfo
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_PopUP
	 	function PKD_Object_PopUp_Consts() {
	 		throw new Error('This is a static class');
	 	}
	 	PKD_Object_PopUp_Consts.EFFECT_DEFAULT = [1.5, true, 0]; //zoom, isUpdateZoom, +toTextSize


	 	class PKD_Object_PopUp
	 	{
	 		constructor(text, colorCss, iconIndex, fontSettings)
	 		{
	 			this._text = text || null;
	 			this._color = colorCss;
	 			this._iconIndex = iconIndex || null;
	 			this._fontSettings = fontSettings || PKD_Object_PopUp.FONT_DEFAULT();
	 			this._effectType = PKD_Object_PopUp_Consts.EFFECT_DEFAULT;
	 			this._sprite = null;
	 		}

	 		getText() {
	 			return this._text;
	 		}

	 		getFontSettings() {
	 			return this._fontSettings;
	 		}

	 		setX(x) {
	 			this.x = x;
	 			this._sprite.x = x;
	 		}

	 		setY(y) {
	 			this.y = y;
	 			this._sprite.y = y;
	 		}

	 		setNumered() //This is number value in this PopUp
	 		{
	 			this._numered = true;
	 		}

	 		isNumered()
	 		{
	 			return (this._numered === true);
	 		}

	 		hasIcon() {
	 			return (this._iconIndex != null);
	 		}

	 		setExtraText(text) {
	 			this._text = (text + " " + this._text);
	 		}

	 		setEffectSettings(settings) {
	 			this._effectType = settings;
	 		}

	 		setup(x, y, width, layer)
	 		{
	 			this._layer = layer;
	 			this._width = width;
	 			this.x = x;
	 			this.y = y;
	 			this._refresh();
	 		}

	 		dispose()
	 		{
	 			this._sprite.bitmap.clear();
	 			this._layer.removeChild(this._sprite);
	 			this._sprite = null;
	 		}

	 		update()
	 		{
	 			if(this._sprite != null) {
	 				this._update_zoom();
	 				this._sprite.update();
	 			}
	 		}

	 		static FONT_DEFAULT() {
	 			return ['Skratch Punk',30, false]; //name, size, italic
	 		}

	 		//PRIVATE
	 		_refresh()
	 		{
	 			let h = 72;
	 			let bitmap = new Bitmap(this._width, h);
	 			bitmap.addLoadListener(function()
	 			{
	 				if(this._fontSettings[0] != null)
	 					bitmap.fontFace = this._fontSettings[0];
	 				bitmap.fontSize = this._fontSettings[1];
	 				bitmap.fontItalic = this._fontSettings[2];
	 				if(this._color) 
	 				{
	 					bitmap.textColor = this._color;
	 				}
	 				else 
	 					bitmap.textColor = ColorCSS.WHITE;
	 			

		 			var dx = 0;
		 			var dw = 0;
		 			var tw = (this._text != null) ?  bitmap.measureTextWidth(this._text) : 0;

		 			while(tw > this._width){
		 				bitmap.fontSize = bitmap.fontSize - 4;
		 				tw = bitmap.measureTextWidth(this._text);
		 			}

		 			if(this._iconIndex) {
		 				dx += 24;
		 				dw += 24;
		 				SDK.drawIcon((dx + ((this._width - tw)/2) - 36), (h - 24)/2, this._iconIndex, bitmap, 24);
		 			}

		 			if(this._text)
		 				bitmap.drawText(this._text, dx + 2, 0, this._width - dw, h, 'center');
	 			}.bind(this));

	 			this._sprite = new Sprite(bitmap);
	 			this._sprite.x = this.x;
	 			this._sprite.y = this.y;
	 			this._sprite.scale.x = this._effectType[0];
	 			this._sprite.scale.y = this._effectType[0];

	 			this._layer.addChild(this._sprite);
	 		}

	 		_update_zoom()
	 		{
	 			if(this._effectType[1]) {
	 				this._sprite.scale.x = Math.max(this._sprite.scale.x - 0.075, 1.0);
	 				this._sprite.scale.y = this._sprite.scale.x;
	 			}
	 			this._sprite.opacity = Math.max(this._sprite.opacity - 2, 0);
	 			if(this._sprite.opacity == 0) {
	 				this._layer.removeChild(this._sprite);
	 				this._sprite = null;
	 			}
	 		}

	 	}
	 	//END PKD_Object_PopUp
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//PKD_Object_PopUpMachine
		function PKD_Object_PopUpMachine_Consts() {
	 		throw new Error('This is a static class');
	 	}
	 	PKD_Object_PopUpMachine_Consts.Y_STEP = 24;
	 	PKD_Object_PopUpMachine_Consts.MAX_TIME = 60;
	 	PKD_Object_PopUpMachine_Consts.SETTINGS = [1, false, 12];

		class PKD_Object_PopUpMachine
		{
			constructor(x, y, width, stack_size, parent)
	 		{
	 			this._x = x;
	 			this._y = y;
	 			this._width = width;
	 			this._stack_size = stack_size;
	 			this._parent = parent;
	 			this._effectType = null;

	 			this._items = [];
	 			this._timers = [];

	 			this._init_items();
	 		}

	 		setEffectSettings(effect) {
	 			this._effectType = effect;
	 		}

	 		push(popUpItem)
	 		{
	 			if(this._effectType != null)
	 				popUpItem.setEffectSettings(this._effectType);

	 			popUpItem.setup(this._x, this._y, this._width, this._parent);

	 			let item = this._items.shift();
	 			if(item != null) item.dispose();

	 			this._items.push(popUpItem);
	 			this._step();
	 			this._timers.shift();
	 			this._timers.push(0);
	 		}

	 		clear()
	 		{
	 			this._items.forEach(function(item){
	 				if(item != null) item.dispose();
	 			});
	 			this._items = [];
	 			this._timers = [];
	 			this._init_items();
	 		}

	 		update()
	 		{
	 			this._update_timers();
	 			this._items.forEach(function(item){
	 				if(item != null) item.update();
	 			});
	 		}

	 		//PRIVATE
	 		_init_items()
	 		{
	 			SDK.times(this._stack_size, function() { 
	 				this._items.push(null);
	 				this._timers.push(null);
	 			}.bind(this));
	 		}

	 		_update_timers()
	 		{
	 			SDK.times(this._stack_size, function(i) { 
	 				var index = (this._timers.length - 1) - i; //Reverse 
	 				var timer = this._timers[index];
	 				if(timer == null)
	 					return;
	 				else
	 				{
	 					if(timer < PKD_Object_PopUpMachine_Consts.MAX_TIME)
	 						this._timers[index] = this._timers[index] + 1;
	 					if(timer == PKD_Object_PopUpMachine_Consts.MAX_TIME) {
	 						if(this._items[index] != null)
	 						{
	 							this._items[index].dispose();
	 						}
	 						this._items[index] = null;
	 						this._timers[index] = null;
	 					}
	 				}
	 			}.bind(this));
	 		}

	 		_step()
	 		{
	 			SDK.times(this._items.length, function(i) {
	 				var index = (this._items.length - 1) - i; //Reverse 
	 				var item = this._items[index];
	 				if(item == null)
	 					return;
	 				let y = this._y + (PKD_Object_PopUpMachine_Consts.Y_STEP * i);
	 				this._items[index].setX(this._x);
	 				this._items[index].setY(y);
	 			}.bind(this));
	 		}
		}

		class PKD_Object_PopUpMachineA extends PKD_Object_PopUpMachine {
			constructor(battler, layer) {
				super(battler.sprite.screenX, 24, 96, 1, layer);
			}

			push(popUpItem)
			{
				popUpItem.setEffectSettings(PKD_Object_PopUpMachine_Consts.SETTINGS);
				PKD_Object_PopUpMachine.prototype.push.call(this, popUpItem);
			}
		}
	 	//END PKD_Object_PopUpMachine
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_Bar
	 	class PKD_Object_Bar
	 	{
	 		constructor(bitmap)
	 		{
	 			this._bitmap = bitmap;
	 			this._rect = undefined;
	 			this._bColor = ColorPKD.BLACK();
	 			this._bColor.prepare();
	 			this._color = ColorPKD.WHITE();
	 			this._isGradient = true;
	 			this._mValue = 0;
	 			this._text_l = null;
	 			this._text_r = null;
	 			this._text_c = null;
	 			this._lValue = -1; //Last value
	 			this._value = 0;

	 			this._calculate_gradient();
	 		}

	 		setPosition(x, y, w, h)
	 		{
	 			this._rect = new Rectangle(x,y,w,h);
	 		}

	 		setValue(value)
	 		{
	 			if(value <= 0)
	 				value = 0;
	 			if(value > this._mValue)
	 				value = this._mValue;
	 			this._value = value;
	 		}

	 		getValue()
	 		{
	 			return this._value;
	 		}

	 		//PARAMS
	 		//color , bColor, maxValue, value
	 		config(params)
	 		{
	 			this._color = params.color || ColorPKD.WHITE();
	 			this._bColor = params.bColor || ColorPKD.BLACK();
	 			this._mValue = params.maxValue || 0;
	 			this.setValue(params.value || 0);
	 		}

	 		setText(text, position)
	 		{
	 			switch(position)
	 			{
	 				case 'center':
	 					this._text_c = text;
	 					break;
	 				case 'left':
	 					this._text_l = text;
	 					break;
	 				case 'right':
	 					this._text_r = text;
	 					break;
	 			}
	 		}

	 		allowGradient(isAllowed)
	 		{
	 			isAllowed = SDK.check(isAllowed, true);
	 			this._isGradient = isAllowed;
	 			if(this._isGradient)
	 				this._calculate_gradient();
	 		}

	 		update()
	 		{
	 			if(this._rect === undefined)
	 				return;
	 			if(this._lValue == this._value)
	 				return; //No drawing if not changes
	 			this._draw_back_bar();
	 			if(this._mValue != 0)
	 				this._draw_main_bar();
	 			this._lValue = this._value;
	 		}


	 		//PRIVATE
	 		_draw_back_bar()
	 		{
	 			this._bitmap.fillRect(this._rect.x, this._rect.y, this._rect.width, this._rect.height, this._bColor.toCSS());
	 		}

	 		_draw_main_bar()
	 		{
	 			let width = Math.floor((100 * this._value / this._mValue) * (this._rect.width - 2)/100);

	 			if(this._isGradient) {
	 				this._bitmap.gradientFillRect(this._rect.x + 1, this._rect.y + 1, width, this._rect.height - 2,
	 												 this._color.toCSS(), this._gColor.toCSS(), false);
	 			}
	 			else {
	 				this._bitmap.fillRect(this._rect.x + 1, this._rect.y + 1, width, this._rect.height - 2, this._color.toCSS());
	 			}

	 			let size = this._bitmap.fontSize;
	 			this._bitmap.fontFace = BattleSystem_Consts.FONT;
	 			this._bitmap.fontSize = this._rect.height - 2;

	 			if(this._text_c != null)
	 				this._bitmap.drawText(this._text_c, this._rect.x + 2, this._rect.y, this._rect.width - 4, this._rect.height,'center');
	 			if(this._text_l != null)
	 				this._bitmap.drawText(this._text_l, this._rect.x + 4, this._rect.y, this._rect.width - 8, this._rect.height,'left');
	 			if(this._text_r != null)
	 				this._bitmap.drawText(this._text_r, this._rect.x + 2, this._rect.y, this._rect.width - 6, this._rect.height,'right');
	 			this._bitmap.fontSize = size;
	 		}

	 		_calculate_gradient()
	 		{
	 			this._gColor = this._color.getLightestColor(230);
	 			this._gColor.prepare();
	 		}


	 	}

	 	class PKD_Object_BarHP extends PKD_Object_Bar
	 	{
	 		constructor(bitmap, battler)
	 		{
	 			super(bitmap);
	 			this._battler = battler;
	 			this._color = ColorPKD.RED();
	 			this._color.prepare();
	 			this._mValue = this._battler.mhp;
	 			this._value = this._battler.hp;

	 			this._text_l = TextManager.hpA;
	 			this._text_r = this._value;
	 		}

	 		update()
	 		{
	 			if(this._mValue != this._battler.mhp)
	 			{
	 				this._lValue = -1;
	 				this._mValue = this._battler.mhp;
	 			}
	 			this._value = this._battler.hp;
	 			this._text_r = this._value;
	 			PKD_Object_Bar.prototype.update.call(this); 			
	 		}
	 	}

	 	class PKD_Object_BarHPE extends PKD_Object_Bar
	 	{
	 		constructor(bitmap, battler)
	 		{
	 			super(bitmap);
	 			this._battler = battler;
	 			this._color = ColorPKD.RED();
	 			this._color.prepare();
	 			this._mValue = this._battler.mhp;
	 			this._value = this._battler.hp;
	 			this._text_c = this._get_value();
	 		}

	 		update()
	 		{
	 			if(this._mValue != this._battler.mhp)
	 			{
	 				this._lValue = -1;
	 				this._mValue = this._battler.mhp;
	 			}
	 			this._value = this._battler.hp;
	 			if(this._value != this._lValue)
	 			{
	 				this._text_c = this._get_value();
	 			}
	 			PKD_Object_Bar.prototype.update.call(this); 			
	 		}

	 		//PRIVATE
	 		_get_value()
	 		{
	 			let temp = Math.floor((this._value * 100)/this._mValue);
	 			if(temp <= 0) temp = 1;
	 			return (temp + '%');
	 		}
	 	}

	 	class PKD_Object_BarMP extends PKD_Object_Bar
	 	{
	 		constructor(bitmap, battler)
	 		{
	 			super(bitmap);
	 			this._battler = battler;
	 			this._color = ColorPKD.BLUE();
	 			this._color.prepare();
	 			this._mValue = this._battler.mmp;
	 			this._value = this._battler.mp;

	 			this._text_l = TextManager.mpA;
	 			this._text_r = this._value;
	 		}

	 		update()
	 		{
	 			if(this._mValue != this._battler.mmp)
	 			{
	 				this._lValue = -1;
	 				this._mValue = this._battler.mmp;
	 			}
	 			this._value = this._battler.mp;
	 			this._text_r = this._value;
	 			PKD_Object_Bar.prototype.update.call(this); 		
	 		}
	 	}

	 	class PKD_Object_BarTP extends PKD_Object_Bar
	 	{
	 		constructor(bitmap, battler)
	 		{
	 			super(bitmap);
	 			this._battler = battler;
	 			this._color = ColorPKD.GREEN();
	 			this._color.prepare();
	 			this._mValue = this._battler.maxTp();
	 			this._value = this._battler.tp;

	 			this._text_l = TextManager.tpA;
	 			this._text_r = this._value;
	 		}

	 		update()
	 		{
	 			this._value = this._battler.tp;
	 			this._text_r = this._value;
	 			PKD_Object_Bar.prototype.update.call(this); 	 					
	 		}
	 	}

	 	class PKD_Object_BarWT extends PKD_Object_Bar
	 	{
	 		constructor(bitmap, battler)
	 		{
	 			super(bitmap);
	 			this._timer = battler.getTimer(); 
	 			this._color = ColorPKD.MAGENTA();
	 			this._color.prepare();
	 			this._text_r = this._get_value();
	 			this._text = BattleSystem_Consts.STRING_TIMER_READY[SDK.isRU()];
	 			this._miniInfo = false;
	 		}

	 		setMiniInfo()
	 		{
	 			this._miniInfo = true;
	 			this._text_r = null;
	 			this._text_c = null;
	 		}

	 		update()
	 		{
	 			this._mValue = this._timer.getMaxValue();
	 			this._value = this._timer.getValue();
	 			if(this._timer.isReady() && !this._miniInfo)
	 			{
	 				this._text_r = null;
	 				this._text_c = this._text;
	 			} else {
	 				this._text_r = this._get_value();
	 				this._text_c = null;
	 			}
	 			PKD_Object_Bar.prototype.update.call(this); 			
	 		}

	 		//PRIVATE
	 		_get_value()
	 		{
	 			let value = (this._timer.getMaxValue() - this._timer.getValue())/60;
	 			return value.toFixed(1);
	 		}
	 	}
	 	//END PKD_Object_Bar
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_KillInfo
	 	function PKD_Object_KillInfo_Consts() {
	 		throw new Error('This is a static class');
	 	}
	 	PKD_Object_KillInfo_Consts.Y_STEP = 25;
	 	PKD_Object_KillInfo_Consts.MAX_TIME = 15; //This is run on another Thread

	 	class PKD_Object_KillInfo
	 	{
	 		constructor(x, y, width, stack_size, parent)
	 		{
	 			this._x = x;
	 			this._y = y;
	 			this._width = width;
	 			this._stack_size = stack_size;
	 			this._parent = parent;

	 			this._items = [];
	 			this._timers = [];

	 			this._init_items();
	 		}

	 		push(user, target)
	 		{
	 			let item = this._items.shift();
	 			if(item != null) 
	 				this._parent.removeChild(item);

	 			let sprite = this._create_info_sprite(user, target);
	 			this._items.push(sprite);
	 			this._step();
	 			this._timers.shift();
	 			this._timers.push(0);
	 		}

	 		clear()
	 		{
	 			this._items.forEach(function(item){
	 				if(item != null) this._parent.removeChild(item);
	 			}.bind(this));
	 			this._items = [];
				this._timers = [];
				this._init_items();
	 		}

	 		hide()
	 		{
	 			this._items.forEach(function(item){
	 				if(item != null) item.visible = false;
	 			});
	 		}

	 		show()
	 		{
	 			this._items.forEach(function(item){
	 				if(item != null) item.visible = true;
	 			});
	 		}

	 		update()
	 		{
	 			this._update_timers();
	 			this._items.forEach(function(item){
	 				if(item != null) item.update();
	 			});
	 		}


	 		//PRIVATE
	 		_init_items()
	 		{
	 			SDK.times(this._stack_size, function() { 
	 				this._items.push(null);
	 				this._timers.push(null);
	 			}.bind(this));
	 		}

	 		_update_timers()
	 		{
	 			SDK.times(this._stack_size, function(i) { 
	 				var index = (this._timers.length - 1) - i; //Reverse 
	 				var timer = this._timers[index];
	 				if(timer == null)
	 					return;
	 				else
	 				{
	 					if(timer < PKD_Object_KillInfo_Consts.MAX_TIME)
	 						this._timers[index] = this._timers[index] + 1;
	 					if(timer == PKD_Object_KillInfo_Consts.MAX_TIME) {
	 						if(this._items[index] != null)
	 							this._parent.removeChild(this._items[index]);
	 						this._items[index] = null;
	 						this._timers[index] = null;
	 					}
	 				}
	 			}.bind(this));
	 		}

	 		_step()
	 		{
	 			SDK.times(this._items.length, function(i) {
	 				var index = (this._items.length - 1) - i; //Reverse 
	 				var item = this._items[index];
	 				if(item == null)
	 					return;
	 				let y = this._y + (PKD_Object_KillInfo_Consts.Y_STEP * i);
	 				this._items[index].x = this._x;
	 				this._items[index].y = y;
	 			}.bind(this));
	 		}

	 		_create_info_sprite(user, target)
	 		{
	 			let sprite = new Sprite();
	 			sprite.x = this._x;
	 			sprite.y = 0;
	 			let bitmap = new Bitmap(this._width, 24);
	 			sprite.bitmap = bitmap;
	 			bitmap.addLoadListener(function(){
	 				let dx = this._draw_info_line(0, user, sprite);
	 				SDK.drawIcon(dx + 1, 0, 'attack', bitmap);
	 				dx += 30;
	 				this._draw_info_line(dx, target, sprite);
	 			}.bind(this));

	 			this._parent.addChild(sprite);
	 			return sprite;
	 		}

	 		_draw_info_line(x, battler, sprite)
	 		{
	 			var color = battler.isEnemy() ? ColorCSS.RED : ColorCSS.GREEN;
	 			if(battler.isPlayer()) {
	 				color = ColorCSS.AQUA;
	 			} 
	 			sprite.bitmap.textColor = color;
	 			sprite.bitmap.drawText(battler.name(), x, 0, this._width - 78 + x, 24);
	 			x += sprite.bitmap.measureTextWidth(battler.name());
	 			return x;
	 		}

	 	}
	 	//END PKD_Object_KillInfo
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_RechargeBar
	 	class PKD_Object_RechargeBar
	 	{
	 		constructor(x, y, height, layer, battler) {
	 			this._x = x;
	 			this._y = y;
	 			this._height = height;
	 			this._layer = layer;
	 			this._battler = battler;
	 			this._draw = false;
	 			this._fores = [];

	 			let bitmap = new Bitmap(30, height);
	 			this._sprite = new Sprite(bitmap);
	 			this._sprite.x = this._x;
	 			this._sprite.y = this._y;
	 			this._layer.addChild(this._sprite);

	 			this._fill_fores();
	 			this.refresh();
	 		}

	 		setDrawMode(isDraw) {
	 			this._draw = isDraw;
	 			this.refresh();
	 		}

	 		update()
	 		{
	 			this._sprite.update();
	 			this.refresh();
	 		}

	 		refresh()
	 		{
	 			this._fores.forEach(function(item) {
	 				item.update();
	 				item.bitmap.clear();
	 			});
	 			this._sprite.bitmap.clear();
	 			//this._sprite.bitmap.fillRect(0,0,30,this._height,ColorPKD.RED().toCSS());

	 			if(this._draw) {
	 				this._drawAll();
	 			}
	 		}

	 		//PRIVATE
	 		_fill_fores() {
	 			SDK.times(4, function(i) {
	 				let bitmap = new Bitmap(24,24);
	 				let sprite = new Sprite(bitmap);
	 				sprite.x = 6;
	 				sprite.y = this._yVal(i);
	 				sprite.opacity = 100;
	 				this._sprite.addChild(sprite);
	 				this._fores.push(sprite);
	 			}.bind(this));
	 		}

	 		_drawAll() {
	 			let skills = this._battler.getRechargeSkills();
	 			var icons = [];
	 			var speeds = [];
	 			var times = [];
	 			for(let key in skills) {
					if(skills.hasOwnProperty(key)) {
						let item = $dataSkills[key];
						icons.push(item.iconIndex);
	 					speeds.push(item.speed);
	 					times.push(skills[key]);
					}
				}
	 			for(let i = 0; i<icons.length; i++) {
	 				SDK.drawIcon(6, this._yVal(i), icons[i], this._sprite.bitmap, 24);
	 			}
	 			for(let i = 0; i<times.length; i++) {
	 				this._drawTimeBar(i, this._calc_px_percent(times[i], speeds[i]));
	 			}
	 		}

	 		_calc_px_percent(c,m) {
	 			var c_inp = (100* c) / (m * 60);
	 			return Math.floor((24 * c_inp) / 100);
	 		}

	 		_drawTimeBar(index, height) {
	 			var color = (height < 16) ? ((height < 8) ? ColorCSS.GREEN : ColorCSS.YELLOW ) : ColorCSS.RED;
	 			this._fores[index].bitmap.fillRect(0, height, 24, 24 - height, color);
	 		}

	 		_yVal(index) {
	 			return (PKD_Object_KillInfo_Consts.Y_STEP * index);
	 		}

	 	}
	 	//END PKD_Object_RechargeBar
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_StatusBar
	 	class PKD_Object_StatusBar
	 	{
	 		constructor(x, y, layer, battler)
	 		{
	 			this._layer = layer;
	 			this._battler = battler;

	 			this._draw = false;
	 			this._isDrawTimes = true;
	 			this._limit = 0; //Unlimited

	 			//[[index, time in secs],...]
	 			this._times = [];
	 			this._timesB = [];

	 			let bitmap = new Bitmap(80, this._layer.height);
	 			this._sprite = new Sprite(bitmap);
	 			this._sprite.x = x;
	 			this._sprite.y = y;
	 			this._layer.addChild(this._sprite);

	 			this.refresh();
	 		}

	 		setNoDrawTime() {
	 			this._isDrawTimes = false;
	 		}

	 		setLimit(limit) {
	 			this._limit = limit;
	 		}

	 		setDrawMode(isDraw) {
	 			this._draw = isDraw;
	 			this.refresh();
	 		}

	 		update() {
	 			this.refresh();
	 		}

	 		refresh() {
	 			this._sprite.bitmap.clear();
	 			//this._sprite.bitmap.fillRect(0,0,80,this._layer.height - 26, ColorPKD.RED().toCSS());
	 			if(this._draw == true) {
	 				this._drawIcons();
	 				if(this._isDrawTimes == true) {
	 					this._drawTimes();
	 				}
	 			}
	 		}

	 		//PRIVATE
	 		_drawIcons() {
	 			let icons = this._battler.allIcons();
	 			if(this._limit == 0) {
	 				for(let i = 0; i<icons.length; i++) {
	 					SDK.drawIcon(0, this._yVal(i), icons[i], this._sprite.bitmap, 24);
	 				}
	 			} else {
	 				for(let i = 0; i<this._limit; i++) {
	 					SDK.drawIcon(0, this._yVal(i), icons[i], this._sprite.bitmap, 24);
	 				}
	 			}
	 		}

	 		_drawTimes() {
	 			this._drawStatesTimes();
	 			this._drawBuffTimes();
	 		}

	 		_drawStatesTimes() {
	 			let states = this._battler.states();
	 			let stateTimes = this._battler._stateTurns;
	 			this._sCount = states.length;
	 			this._times = [];
	 			var time = '';
	 			for(let i = 0; i<states.length; i++) {	
	 				let state = states[i];
	 				let time = stateTimes[state.id];
	 				if(state.autoRemovalTiming == 1) {
	 					this._times.push([i, ' :A']);
	 				} else {
	 					if(time > 0) {
	 						this._times.push([i, this._framesToTime(time)]);
	 					}
	 				}
	 			}
	 			for(let i = 0; i<this._times.length; i++) {
	 				let item = this._times[i];
	 				this._drawTime(item[0], item[1]);
	 			}
	 		}

	 		_drawBuffTimes() {
	 			let buffs = this._battler._buffs;
	 			let bTimes = this._battler._buffTurns;
	 			this._timesB = [];
	 			var ind = 0;
	 			for(let i = 0; i<buffs.length; i++) {
	 				let time = bTimes[i];
	 				if(time) {
	 					this._timesB.push([ind + this._sCount, this._framesToTime(time)]);
	 					ind += 1;
	 				}
	 			}
	 			for(let i = 0; i<this._timesB.length; i++) {
	 				let item = this._timesB[i];
	 				this._drawTime(item[0], item[1]);
	 			}
	 		}

	 		_framesToTime(frames) {
	 			var secs = Math.floor((frames + BattleSystem_Consts.FRAMES) / BattleSystem_Consts.FRAMES);
	 			var string = '';
	 			if(secs > 59) {
	 				var min = Math.floor(secs/60);
	 				var minm = (min < 10) ? ('0' + min) : min;
	 				var secx = secs - (min * 60);
	 				secx = (secx < 10) ? ('0' + secx) : secx;
	 				string = min + ":" + secx;
 	 			} else {
	 				string = '0:' + ((secs < 10) ? ('0' + secs) : secs);
	 			}
	 			return string;
	 		}

	 		_drawTime(index, time) {
	 			this._sprite.bitmap.drawText(time, 28, this._yVal(index), 44, 24, 'center');
	 		}

	 		_yVal(index) {
	 			return this._sprite.height - (PKD_Object_KillInfo_Consts.Y_STEP * (index + 1));
	 		}
	 	}
	 	//END PKD_Object_StatusBar
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_ReadyBar
	 	class PKD_Object_ReadyBar
	 	{
	 		constructor(x, y, width, battler, layer) {
	 			this._battler = battler;
	 			this._inWait = false;
	 			let bitmap = new Bitmap(width, 24);
	 			this._sprite = new Sprite(bitmap);
	 			this._sprite.x = x;
	 			this._sprite.y = y;
	 			layer.addChild(this._sprite);
	 			this.update();
	 		}

	 		hide() {
	 			this._sprite.visible = false;
	 		}

	 		show() {
				this._sprite.visible = true;
	 		}

	 		update()
	 		{
	 			if(this._sprite.visible == false)
	 				return;
	 			if(this._battler.isReadyForAction()) {
	 				this._inWait = false;
	 				this._sprite.bitmap.clear();

	 				let skills = this._battler.getBattleSkills();
					var temp = [];
					for(let i = 0; i<skills.length; i++) {
						if(this._battler.canUse(skills[i])) {
							temp.push(skills[i].iconIndex);
						}
					}

					if(temp.length != 0) {
						for(let i = 0; i<temp.length; i++) {
							SDK.drawIcon(24 * i, 0, temp[i], this._sprite.bitmap);
						}
					}

	 			} else {
	 				if(!this._inWait) {
	 					this._sprite.bitmap.clear();
	 					//this._sprite.fontSize = 8;
	 					this._sprite.bitmap.drawText(BattleSystem_Consts.STRING_WAIT_TURN[SDK.isRU()], 4, 0, this._sprite.bitmap.width - 8 , 20, 'center');
	 					this._inWait = true;
	 				}
	 			}
	 		}
	 	}
	 	//ENDPKD_Object_ReadyBar
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//PKD_Object_Timer
	 	class PKD_Object_Timer
	 	{
	 		constructor()
	 		{
	 			this._ready = false;
	 			this._paused = false;
	 			this._mValue = 0;
	 			this._value = 0;
	 		}

	 		getMaxValue()
	 		{
	 			return this._mValue;
	 		}

	 		getValue()
	 		{
	 			return this._value;
	 		}

	 		setMaxTime(frameCount)
	 		{
	 			frameCount = Math.abs(Math.round(frameCount));
	 			this._mValue = frameCount;
	 			if(this._value > this._mValue)
	 				this._value = this._mValue;
	 		}

	 		reset()
	 		{
	 			this._value = 0;
	 			this._ready = false;
	 		}

	 		isReady()
	 		{
	 			return this._ready;
	 		}

	 		start(frameCount)
	 		{
	 			this._value = 0;
	 			this._mValue = Math.abs(Math.round(frameCount));
	 			this._ready = false;
	 			this._paused = false;
	 		}

	 		stop()
	 		{
	 			this.start(0);
	 		}

	 		pause()
	 		{
	 			if(this._paused)
	 				return;
	 			if(this._mValue == 0)
	 				return;
	 			this._paused = true;
	 		}

	 		resume()
	 		{
	 			this._paused = false;
	 		}

	 		update()
	 		{
	 			if(!this._ready)
	 			{
	 				if(!this._paused)
	 				{
	 					if(this._value < this._mValue)
	 						this._value += 1;
	 				}
	 				this._ready = (this._value >= this._mValue);
	 			}
	 		}

	 	}
	 	//END PKD_Object_Timer
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Game_BattlerBase
	 	let pkd_GameBattlerBase_initMembers = Game_BattlerBase.prototype.initMembers;
	 	Game_BattlerBase.prototype.initMembers = function() {
	 		pkd_GameBattlerBase_initMembers.call(this);
	 		this._popups = [];
	 	}

	 	let pkd_GameBattlerBase_eraseState = Game_BattlerBase.prototype.eraseState;
	 	Game_BattlerBase.prototype.eraseState = function(stateId) {
	 		if(this._states.include(stateId)) {
	 			PKD_PopInfoManager.makeStatePopUp(this, stateId, true);
	 		}
	 		pkd_GameBattlerBase_eraseState.call(this, stateId);		
	 	}

	 	//NEW
	 	Game_BattlerBase.prototype.getInfoPops = function() {
	 		return this._popups;
	 	}

	 	Game_BattlerBase.prototype.addInfoPop = function(info) {
	 		this._popups.push(info);
	 	}

	 	Game_BattlerBase.prototype.clearInfoPops = function() {
	 		this._popups = [];
	 	}

		Game_BattlerBase.prototype.isPlayer = function() {
			return false;
		}	 	

	 	//END Game_BattlerBase
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
	 	//Game_Battler
	 	let pkd_GameBattler_initMembers = Game_Battler.prototype.initMembers;
	 	Game_Battler.prototype.initMembers = function() {
	 		pkd_GameBattler_initMembers.call(this);
	 		this._waitTime = BattleSystem_Consts.MINIMUM_TURN_WAIT; //Время ожидания хода
	 		this._fastBattle = false;
	 		this._timer = new PKD_Object_Timer();
	 		this._insWtKoef = 1.0; //Дополнительный коэффициент времени хода
	 		this._splashDmgKoef = 0; //Коэффициент урона по площади

	 	}

	 	Game_Battler.prototype.getTimer = function()
	 	{
	 		return this._timer;
	 	}

	 	Game_Battler.prototype.isReadyForAction = function()
	 	{
	 		return this._timer.isReady();
	 	}

	 	Game_Battler.prototype.updateInBattle = function()
	 	{
	 		this._timer.update();
	 		this.updateStateTurns();
	 		this.updateBuffTurns();
	 		this.removeStatesAuto(2);
	 		this.removeBuffsAuto();
	 	}

	 	Game_Battler.prototype.resetWait = function()
	 	{
	 		this._calc_wait_time();
	 		this._timer.reset();
	 	}

	 	Game_Battler.prototype.setFastBattle = function()
	 	{
	 		this._fastBattle = true;
	 		this._calc_wait_time();
	 	}

	 	Game_Battler.prototype.setSplashDmg = function(dmg)
	 	{
	 		this._splashDmgKoef = dmg;
	 	}

	 	Game_Battler.prototype._calc_wait_time = function()
	 	{
	 		var speed = BattleSystem_Consts.MINIMUM_TURN_WAIT + ((((this.def+this.mdf)/2)/this.agi) * BattleSystem_Consts.BATTLE_TURN_STEP);
	 		speed = speed * this._insWtKoef;

	 		var attackSpeed = this.attackSpeed();
	 		if(attackSpeed > 0)
	 			speed -= this.attackSpeed();
	 		else
	 			speed += this.attackSpeed();
	 		speed = Math.max(BattleSystem_Consts.MINIMUM_TURN_WAIT, speed);
	 		this._waitTime = speed;
	 		if(this._fastBattle == true) {
	 			this._waitTime = speed * BattleSystem_Consts.BATTLE_FAST_KOEF;
	 		}
	 		this._timer.setMaxTime(this._waitTime);
	 	}

	 	//Override
	 	let pkd_GameBattler_onBattleStart = Game_Battler.prototype.onBattleStart;
	 	Game_Battler.prototype.onBattleStart = function() {
	 		pkd_GameBattler_onBattleStart.call(this); 
		   	this._timer = new PKD_Object_Timer();
		   	this._fastBattle = false;
		   	this._calc_wait_time();
		   	//console.log(this.name() + " on Battle: " + this._timer.getMaxValue());
		};

		let pkd_GameBattler_onBattleEnd = Game_Battler.prototype.onBattleEnd;
	 	Game_Battler.prototype.onBattleEnd = function() {
	 		pkd_GameBattler_onBattleEnd.call(this);
	 		this.clearInfoPops();
	 	}


		let pkd_GameBattler_canMove = Game_Battler.prototype.canMove;
		Game_Battler.prototype.canMove = function() {
			var old = pkd_GameBattler_canMove.call(this);
			if($gameParty.inBattle())
		    	return (old && this.isReadyForAction());
		    else return old;
		};

		Game_Battler.prototype.onTurnEnd = function() {
		    this.regenerateAll();
		};

		Game_Battler.prototype.onAllActionsEnd = function() {
		    this.clearResult();
		    this.removeStatesAuto(1); 
		    this.resetWait();
		    this._insWtKoef = 1.0;
		};

		Game_Battler.prototype.resetStateCounts = function(stateId) {
		    var state = $dataStates[stateId];
		    var variance = 0;
		    if(state.autoRemovalTiming != 1) {
		    	 //For now, turns calcs in a seconds
		    	variance += Math.max(state.maxTurns - state.minTurns, 0);
		    	this._stateTurns[stateId] = (state.minTurns + Math.randomInt(1 + variance)) * BattleSystem_Consts.FRAMES;
		    	if(this._fastBattle)
		    		this._stateTurns[stateId] *= BattleSystem_Consts.BATTLE_FAST_KOEF;
		    } else {
		    	this._stateTurns[stateId] = 1; //After Action
		    }
		};

		Game_Battler.prototype.overwriteBuffTurns = function(paramId, turns) {
			var temp = turns * BattleSystem_Consts.FRAMES;
			if(this._fastBattle)
				temp *= BattleSystem_Consts.BATTLE_FAST_KOEF; 

		    if(this._buffTurns[paramId] < temp) {
		        this._buffTurns[paramId] = temp;
		    }
		};

		let pkd_GameBattler_useItem = Game_Battler.prototype.useItem;
		Game_Battler.prototype.useItem = function(item) {
			if($gameParty.inBattle())
			{
				let i = new Game_Item(item);
				if(i.isInstantCast()) {
					this._insWtKoef = 0;
				} else {
					this._insWtKoef = i.timeMod;
				}
			}
		    pkd_GameBattler_useItem.call(this, item);
		};

		Game_Battler.prototype.addNewState = function(stateId) {
			Game_BattlerBase.prototype.addNewState.call(this, stateId);
			if(this._states.include(stateId))
				PKD_PopInfoManager.makeStatePopUp(this, stateId, false);
		}

		let pkd_GameBattler_addBuff = Game_Battler.prototype.addBuff;
		Game_Battler.prototype.addBuff = function(paramId, turns) {
			PKD_PopInfoManager.makeBuffPopUp(this, paramId, true);
			pkd_GameBattler_addBuff.call(this, paramId, turns);
		}

		let pkd_GameBattler_addDebuff = Game_Battler.prototype.addDebuff;
		Game_Battler.prototype.addDebuff = function(paramId, turns) {
			PKD_PopInfoManager.makeBuffPopUp(this, paramId, false);
			pkd_GameBattler_addDebuff.call(this, paramId, turns);
		}

		let pkd_GameBattler_regenerateAll = Game_Battler.prototype.regenerateAll;
		Game_Battler.prototype.regenerateAll = function() {
			this.clearResult();
			pkd_GameBattler_regenerateAll.call(this);
			if(this.isAlive())
				PKD_PopInfoManager.makeDamagePopUp(this);
		}

	 	Object.defineProperty(Game_Battler.prototype, 'sprite', {
		    get: function() {
		    	if(this.isEnemy())
		        	//return BattleManagerRTBS.getSpriteset().getEnemySprite(this.index());
		        	return this._sprite;
		        else 
		        	return BattleManagerRTBS.getSpriteset().getActorSprite(this.index());
		    },
		    configurable: true
		});
		//END Game_Battler
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Game_Actor
		Game_Actor.prototype.isSpriteVisible = function() {
	    	return true;
		};

		let pkd_GameActor_resetStateCounts = Game_Actor.prototype.resetStateCounts;
		Game_Actor.prototype.resetStateCounts = function(stateId) {
		    pkd_GameActor_resetStateCounts.call(this, stateId);
		    if (this._fastBattle)
		    	this._stateSteps[stateId] *= BattleSystem_Consts.BATTLE_FAST_KOEF;

		};

		let pkd_GameActor_initMembers = Game_Actor.prototype.initMembers;
		Game_Actor.prototype.initMembers = function() {
			pkd_GameActor_initMembers.call(this);
			this._battle_skills = [0,0,0,0]; //ID Навыков для круга
			this._battle_skills_set = null;
			this._recharge_skills = {}; //ID навыка -> время до восстановления (если 0 - можно use)
			this._skill_m_counter = 0; //Максимальный счётчик серии навыка
			this._last_counter_skill_id = null; //Последний скилл, для создания цепочки
		}

		let pkd_GameActor_setup = Game_Actor.prototype.setup;
		Game_Actor.prototype.setup = function(actorId) {
			pkd_GameActor_setup.call(this, actorId);
			this._waitMod = $dataActors[actorId].meta.WaitMod || 1; //Коэффициент времени хода
		}

		Game_Actor.prototype.addBattleSkill = function(item, index) {
			this._battle_skills_set = null;
			var isExist = -1;
			for(let i = 0; i<4 ; i++) {
				if(this._battle_skills[i] == item.id) {
					isExist = i;
					break;
				}
			}

			if(isExist != -1) {
				this._battle_skills[isExist] = 0;  
			}

			this._battle_skills[index] = item.id;
		}

		Game_Actor.prototype.getRechargeSkills = function()
		{
			return this._recharge_skills;
		}

		Game_Actor.prototype.getBattleItemsCount = function()
		{
			return this.getBattleItems().length;
		}

		Game_Actor.prototype.hasItems = function()
		{
			return $gameParty.hasBattleItems(); 
		}

		Game_Actor.prototype.hasSkills = function()
		{
			for(let i = 0; i<this._battle_skills.length; i++) {
				if(this._battle_skills[i] > 0)
					return true;
			}
			return false;
		}

		Game_Actor.prototype.canUseSkills = function()
		{
			if(this.hasSkills()) {
				for(let i = 0; i<this._battle_skills.length; i++) {
					var skill = $dataSkills[this._battle_skills[i]];
					if(this.canUse(skill))
						return true;
				}
			}
			return false;
		}

		let pkd_GameActor_usableSkills = Game_Actor.prototype.usableSkills;
		Game_Actor.prototype.usableSkills = function() {
			if($gameParty.inBattle()) {
				var list = [];
				this._battle_skills.forEach(function(skillId){
					if(skillId != 0)
						list.push($dataSkills[skillId]);
				}.bind(this));
				return list.filter(function(skill) {
			        return this.canUse(skill);
			    }, this);
			}
			else
				return pkd_GameActor_usableSkills.call(this);

		    return this.skills().filter(function(skill) {
		        return this.canUse(skill);
		    }, this);
		};

		Game_Actor.prototype.isSkillReady = function(skill_id)
		{
			return (this._recharge_skills[skill_id] === undefined);
		}

		Game_Actor.prototype.startSkillRecharge = function(skill_id, koef)
		{
			if(skill_id == this.guardSkillId())
				return;
			var koef = SDK.check(koef, 1);
			var time = 0;
			if(skill_id != this.attackSkillId()) 
			{	
				time = Math.abs($dataSkills[skill_id].speed) * koef * BattleSystem_Consts.FRAMES;
			} 
			this._recharge_skills[skill_id] = time;
		}

		Game_Actor.prototype.getBattleSkills = function()
		{
			if(this._battle_skills_set == null)
			{
				this._battle_skills_set = [];
				SDK.times(4, function(i){
					if(this._battle_skills[i] > 0) {
						this._battle_skills_set[i] = $dataSkills[this._battle_skills[i]];
					} else {
						this._battle_skills_set[i] = null;
					}
				}.bind(this));
			}
			return this._battle_skills_set;
		}

		Game_Actor.prototype.getBattleItems = function()
		{
			return $gameParty.getBattleItems();
		}

		Game_Actor.prototype.meetsSkillConditions = function(skill) {
			let x = Game_BattlerBase.prototype.meetsSkillConditions.call(this, skill);
			if($gameParty.inBattle())
				return (x && this.isSkillReady(skill.id));
			else return x;
		};

		Game_Actor.prototype.paySkillCost = function(skill) {
		    Game_BattlerBase.prototype.paySkillCost.call(this, skill);
		    //Mulit skill cost pay instantly, for each one
		    if(!$gameParty.inBattle()) return;
		    let temp = new Game_Item(skill); 
		    if(temp.isMultyCounter()) {
		    	if(this._skill_m_counter == 0)
		    		this._last_counter_skill_id = temp.id;
		    	this._skill_m_counter += 1;
		    	if(this._skill_m_counter == temp.multiMod)
		    	{
		    		this._reset_counter_skill();
		    	}
		    }
		    else
		    {
		    	//Multi skill chain has been broken by another one skill (exclude Defense)
		    	if((this._last_counter_skill_id != null) && (skill.id != this.guardSkillId()))
		    	{
		    		this._reset_counter_skill();
		    	}
		    	if(skill.speed > 0)
		    		this.startSkillRecharge(skill.id);
		    }
		};

	 	Game_Actor.prototype.onBattleStart = function() {
	 		Game_Battler.prototype.onBattleStart.call(this); 
	 		this._battle_skills_set = null;
	 		this._battle_skills.forEach(function(id) {
	 			if(id != 0) {
	 				let temp = new Game_Item($dataSkills[id]);
	 				if(temp.isInitialReload())
	 					this.startSkillRecharge(id, temp.LoadMod);
	 			}
	 		}.bind(this));
		};

		Game_Actor.prototype.updateInBattle = function()
		{
			Game_Battler.prototype.updateInBattle.call(this);
			for(let key in this._recharge_skills) {
				if(this._recharge_skills.hasOwnProperty(key)) {
					let value = this._recharge_skills[key];
					if(value > 0) {
						this._recharge_skills[key] -= 1;
					}
					if(this._recharge_skills[key] == 0) {
						PKD_PopInfoManager.makeSkillRechargePopUp(this, key);
						delete this._recharge_skills[key];
					}
				}
			}
		}
		
		Game_Actor.prototype.onBattleEnd = function()
		{
			Game_Battler.prototype.onBattleEnd.call(this);
			this._skill_m_counter = 0;
			this._last_counter_skill_id = null;
			this._timer.reset();
			this._recharge_skills = {};
		}
	 	
	 	Game_Actor.prototype.isPlayer = function()
	 	{
	 		return ($gameParty.leader() == this);
	 	}

	 	Game_Actor.prototype.isAutoBattle = function() {
		    if(this._fastBattle == true)
		    	return true;
		    else
		    	return !this.isPlayer();
		};

		Game_Actor.prototype.performDamage = function() {
		    Game_Battler.prototype.performDamage.call(this);
		    if(this.isPlayer()) {
		    	$gameScreen.startShake(5, 5, 10);
		    }
		    SoundManager.playActorDamage();
		};

		let pkd_GameActor_learnSkill = Game_Actor.prototype.learnSkill;
		Game_Actor.prototype.learnSkill = function(skillId) {
			pkd_GameActor_learnSkill.call(this, skillId);
			var slot = this._get_empty_skill_slot();
			if(slot != -1) {
				this._battle_skills[slot] = skillId;
				this._battle_skills_set = null;
			}
		}

		let pkd_GameActor_forgetSkill = Game_Actor.prototype.forgetSkill;
		Game_Actor.prototype.forgetSkill = function(skillId) {
			pkd_GameActor_forgetSkill.call(this, skillId);
			var index = this._battle_skills.indexOf(skillId);
			if (index >= 0) {
				this._battle_skills[index] = 0;
				this._battle_skills_set = null;
			}
		}

		//PRIVATE
	 	Game_Actor.prototype._calc_wait_time = function()
	 	{
	 		Game_Battler.prototype._calc_wait_time.call(this);
	 		this._waitTime = Math.round(this._waitTime * this._waitMod);
	 		if(this.isAutoBattle()) {
	 			let brainSpeed = SDK.rand(this.agi / 2, this.agi * 1.5);
	 			if(this._fastBattle) {
	 				brainSpeed = brainSpeed * BattleSystem_Consts.BATTLE_FAST_KOEF;
	 			}
	 			this._waitTime = Math.floor(this._waitTime + brainSpeed);
	 			this._timer.setMaxTime(this._waitTime);
	 		} 
	 		else {
	 			this._timer.setMaxTime(this._waitTime);
	 		}
	 	}

	 	Game_Actor.prototype._get_empty_skill_slot = function()
	 	{
	 		for(let i = 0; i<4; i++) {
	 			if(this._battle_skills[i] == 0)
	 				return i;
	 		}

	 		return -1;
	 	}

	 	Game_Actor.prototype._reset_counter_skill = function()
		{
			this._skill_m_counter = 0;
			this.startSkillRecharge(this._last_counter_skill_id);
			this._last_counter_skill_id = null;
		}
	 	//END Game_Actor
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Game_Enemy
		/*let pkd_GameEnemy_initMembers = Game_Enemy.prototype.initMembers;
		Game_Enemy.prototype.initMembers = function()
		{
			pkd_GameEnemy_initMembers.call(this);
		}*/

		let pkd_GameEnemy_setup = Game_Enemy.prototype.setup;
		Game_Enemy.prototype.setup = function(enemyId, x, y) {
			pkd_GameEnemy_setup.call(this, enemyId, x , y);
			this._waitMod = $dataEnemies[enemyId].meta.WaitMod || 1;  //Коэффициент времени хода
		}

		Game_Enemy.prototype.attackAnimationId1 = function() { //NEW
		    return 6; 
		};

		Game_Enemy.prototype.attackAnimationId2 = function() { //NEW
		    return 0;
		};

		let pkd_GameEnemy_transform = Game_Enemy.prototype.transform;
		Game_Enemy.prototype.transform = function(enemyId) {
			pkd_GameEnemy_transform.call(this, enemyId);
			BattleManagerRTBS.getInterface().getEnemy().refreshEnemyInfo(this.index()); 
			this.resetWait();
		}

		Game_Enemy.prototype._calc_wait_time = function()
	 	{
	 		Game_Battler.prototype._calc_wait_time.call(this);
	 		let brainSpeed = SDK.rand(this.agi, this.agi + ((this.def + this.mdf)/2));
	 		if(this._fastBattle) {
	 			brainSpeed = brainSpeed * BattleSystem_Consts.BATTLE_FAST_KOEF;
	 		}
	 		this._waitTime = Math.floor(this._waitTime + brainSpeed);
	 		this._waitTime = Math.round(this._waitTime * this._waitMod);
	 		this._timer.setMaxTime(this._waitTime);
	 	}
	 	//END Game_Enemy
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Game_Party
		let pkd_GameParty_initAllItems = Game_Party.prototype.initAllItems;
		Game_Party.prototype.initAllItems = function() {
			pkd_GameParty_initAllItems.call(this);
			this._battle_items = [0,0,0,0]; //Вещи для круга (ID)
			this._battle_items_set = null;
		}

		Game_Party.prototype.getBattleItems = function()
		{
			if(this._battle_items_set == null)
			{
				this._battle_items_set = [];
				SDK.times(4, function(i){
					if(this._battle_items[i] > 0) {
						this._battle_items_set[i] = $dataItems[this._battle_items[i]];
					} else {
						this._battle_items_set[i] = null;
					}
				}.bind(this));
			}
			return this._battle_items_set;
		}

		Game_Party.prototype.addBattleItem = function(item, index) {
			this._battle_items_set = null;
			var isExist = -1;
			for(let i = 0; i<4 ; i++) {
				if(this._battle_items[i] == item.id) {
					isExist = i;
					break;
				}
			}

			if(isExist != -1) {
				this._battle_items[isExist] = 0;  
			}

			this._battle_items[index] = item.id;
		}

		let pkd_GameParty_loseItem = Game_Party.prototype.loseItem;
		Game_Party.prototype.loseItem = function(item, amount, includeEquip) {
			pkd_GameParty_loseItem.call(this, item, amount, includeEquip);
			this.useBattleItem(item);
		}

		Game_Party.prototype.hasBattleItems = function() {
			for(let i = 0; i<this._battle_items.length; i++) {
				if(this._battle_items[i] > 0)
					return true;
			}
			return false;
		}

		Game_Party.prototype.onBattleStart = function()
		{
			Game_Unit.prototype.onBattleStart.call(this);
			this._battle_items_set = null;
		}

		Game_Party.prototype.useBattleItem = function(item) {
			if(!this.hasItem(item, false)) {
				var index = -1;
				for(let i = 0; i<4; i++) {
					if(this._battle_items[i] == item.id) {
						index = i;
						break;
					}
				}
				if(index != -1) {
					this._battle_items[index] = 0;
					this._battle_items_set = null;
				}
			}
		}
		//END Game_Party
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Game_Action
		let pkd_GameAction_makeDamageValue = Game_Action.prototype.makeDamageValue;
		Game_Action.prototype.makeDamageValue = function(target, critical) {
			var value = pkd_GameAction_makeDamageValue.call(this, target, critical);
			if(target._splashDmgKoef > 0)
			{
				value *= (target._splashDmgKoef / 100);
				target._splashDmgKoef = 0;
			}
			var temp = this.calcElementRate(target); //Dmg Rate
			if(temp != 1) {
				PKD_PopInfoManager.calcRate(temp);
			} 
			return value;
		}

		let pkd_GameAction_executeDamage = Game_Action.prototype.executeDamage;
		Game_Action.prototype.executeDamage = function(target, value) {
			pkd_GameAction_executeDamage.call(this, target, value);
			PKD_PopInfoManager.makeDamagePopUp(target);
			if (this.isDrain()) {
				PKD_PopInfoManager.makeDrainPopUp(this.subject());
			}
		}

		let pkd_GameAction_itemEffectRecoverHp = Game_Action.prototype.itemEffectRecoverHp;
		Game_Action.prototype.itemEffectRecoverHp = function(target, effect) {
			pkd_GameAction_itemEffectRecoverHp.call(this, target, effect);
			PKD_PopInfoManager.makeDamagePopUp(target);
		}

		let pkd_GameAction_itemEffectRecoverMp = Game_Action.prototype.itemEffectRecoverMp;
		Game_Action.prototype.itemEffectRecoverMp = function(target, effect) {
			pkd_GameAction_itemEffectRecoverMp.call(this, target, effect);
			PKD_PopInfoManager.makeDamagePopUp(target);
		}

		let pkd_GameAction_itemEffectGainTp = Game_Action.prototype.itemEffectGainTp;
		Game_Action.prototype.itemEffectGainTp = function(target, effect) {
			pkd_GameAction_itemEffectGainTp.call(this, target, effect);
			PKD_PopInfoManager.makeDamagePopUp(target);
		}

		let pkd_GameAction_executeHpDamage = Game_Action.prototype.executeHpDamage;
		Game_Action.prototype.executeHpDamage = function(target, value) {
			pkd_GameAction_executeHpDamage.call(this, target, value);
			if(value == 0) {
				PKD_PopInfoManager.makeZeroDamagePopUp(target);
			}
		}
		//END Game_Action
	//------------------------------------------------------------------------------

	//------------------------------------------------------------------------------
		//Game_Item
		let pkd_GameItem_initialize = Game_Item.prototype.initialize;
		Game_Item.prototype.initialize = function(item) {
		    pkd_GameItem_initialize.call(this, item);
		    if(item)
		    	this._load_extra();
		};

		Game_Item.prototype._load_extra = function() //NEW
		{
			let temp = this.object();
			if(this.isUsableItem()) {
				this._timeMod = temp.meta.TimeMod || 1;
				this._multiMod = temp.meta.MultiMod || 1;
				this._splashMod = temp.meta.SplashMod || 0;
				this._loadMod = temp.meta.LoadMod || 0;
			}
		}	

		//Мгновенное исполнение?
		Game_Item.prototype.isInstantCast = function() 
		{
			return (this.isUsableItem() && (this._timeMod <= 0));
		}

		//Мульти использование? (Серия, "барабан")
		Game_Item.prototype.isMultyCounter = function()
		{
			return (this.isSkill() && (this._multiMod > 1));
		}

		//Подготовка в начале боя?
		Game_Item.prototype.isInitialReload = function()
		{
			return (this.isSkill() && (this._loadMod > 0));
		}

		//Урон по рядом стоящим?
		Game_Item.prototype.isSplashDamage = function()
		{
			return (this.isUsableItem() && (this._splashMod > 0));
		}

	 	Object.defineProperties(Game_Item.prototype, {
			timeMod : { get: function() {return this._timeMod; }, configurable: true },
			multiMod : { get: function() {return this._multiMod; }, configurable: true },
			splashMod : { get: function() {return this._splashMod; }, configurable: true },
			loadMod : { get: function() {return this._loadMod; }, configurable: true },
			id : {get: function() {return this.this._itemId; }, configurable: true}
		});

		//END Game_Item
	//------------------------------------------------------------------------------

	 //TEST ZONE
	 class Test_Circle extends PKD_Scene_Test
	 {

 		create_test()
	 	{
	 		this.mode = 'click'; //'disable', 'select', 'hide'
	 		this.isDisabled = false;
	 		this.isHided = false;
	 		this.circle = new PKD_Object_InputCircle(200, 200, this);
	 		this.circle.setHelps(['Test help','Right help','Down help','Left help']);
	 		//this.circle.setIcons([64,66,67,68]);
	 		this.circle.open();

	 		SDK.times(4, function(i) {
	 			this.circle.addListener(i, function() { 
	 				this.circle.click(i);
	 			}.bind(this));
	 		}.bind(this));

	 	}	

	 	update_test()
	 	{
	 		let index = -1;

	 		if(Input.isTriggered(IKey.W()))
	 			index = 0;

	 		if(Input.isTriggered(IKey.A()))
	 			index = 3;

	 		if(Input.isTriggered(IKey.S()))
	 			index = 2;

	 		if(Input.isTriggered(IKey.D()))
	 			index = 1;

	 		if(index != -1)
	 		{
	 			switch(this.mode)
	 			{
	 				case 'click':
	 					this.circle.click(index);
	 					break;
	 				case 'disable':
	 					this.isDisabled = !this.isDisabled;
	 					this.circle.disableIcon(index, this.isDisabled);
	 					break;
	 				case 'select':
	 					this.circle.select(index);
	 					break;
	 				case 'hide':
	 					this.isHided = !this.isHided;
	 					this.circle.hideSegment(index, this.isHided);
	 					break;
	 			}
	 		}

	 		if(Input.isTriggered(IKey.TAB()))
	 		{
	 			if(this.circle.isOpen()) {
	 				this.circle.close();
	 			}
	 			else {
	 				this.circle.open();
	 			}
	 		}

	 		if(Input.isPressed(IKey.SPACE()))
	 		{
	 			this.circle.showHelp();
	 		} 
	 		else
	 			this.circle.hideHelp();

	 		if(Input.isTriggered('1'))
	 			this.mode = 'click';

	 		if(Input.isTriggered('2'))
	 			this.mode = 'select';

	 		if(Input.isTriggered('3'))
	 			this.mode = 'disable';

	 		if(Input.isTriggered('4'))
	 			this.mode = 'hide';

	 		this.circle.update();
	 	}
	 }

	 class Test_Bar extends PKD_Scene_Test
	 {

 		create_test()
	 	{
	 		this.sprite = new Sprite(new Bitmap(100,40));
	 		this.sprite.x = 100;
	 		this.sprite.y = 100;
	 		this.bar = new PKD_Object_Bar(this.sprite.bitmap);
	 		this.bar.setPosition(0,0,90,30);
	 		this.bar.config({color : ColorPKD.RED(), maxValue : 100, value : 100});
	 		this.bar.setText(TextManager.hpA, 'left');
	 		this.bar.setText(this.bar.getValue(), 'right');	
	 		this.addChild(this.sprite);

	 		this.killInfo = new PKD_Object_KillInfo(200,200,300,4, this);

	 		this.hero = $gameParty.leader();
	 		this.ally = $gameParty.members()[1];
	 		var enemyObject = $dataTroops[1].members[0];
	 		this.enemy = $dataEnemies[enemyObject.enemyId];
	 		this.enemy2 = new Game_Enemy(enemyObject.enemyId, 0, 0);
	 		console.log(this.hero.name());
	 		//console.log(this.ally.name());
	 		console.log(this.enemy.name);
	 		console.log(this.enemy2.hp);

	 			this.createWindowLayer();
	 		    this._messageWindow = new Window_Message();
			    this.addWindow(this._messageWindow);
			    this._messageWindow.subWindows().forEach(function(window) {
			        this.addWindow(window);
			    }, this);

			 this.wl = new WindowLayer();
			 this.wl.move(300,200,100,100);
			 var bitmap = new Bitmap(100,100);
			 bitmap.addLoadListener(function(){
 				bitmap.fillRect(0,0,100,100,ColorPKD.RED().toCSS());
			 });
			
			 var sprite = new Sprite(bitmap);
			 this.wl.addChild(sprite);
			 this.addChild(this.wl);


	 	}	

	 	update_test()
	 	{
	 		this.wl.update();
	 		this.bar.setText(this.bar.getValue(), 'right');
	 		this.sprite.update();
	 		this.bar.update();
	 		this.killInfo.update();

	 		if(Input.isTriggered(IKey.W()))
	 		{
	 			this.killInfo.push(this.hero, this.enemy2);
	 		}
	 		if(Input.isTriggered(IKey.A()))
	 		{
	 			this.killInfo.push(this.enemy2, this.hero);
	 		}
	 		if(Input.isTriggered(IKey.S()))
	 		{
	 			this.killInfo.push(this.ally, this.enemy2);
	 		}
	 		if(Input.isTriggered(IKey.D()))
	 		{

	 		}


	 		if(Input.isTriggered(IKey.SPACE()))
	 		{
				$gameMessage.add('test');
	 		}

	 		if(!$gameMessage.isBusy())
	 		{
		 		if(Input.isTriggered('1'))
		 			this.bar.setValue(this.bar.getValue() - 10);

		 		if(Input.isTriggered('2'))
		 			this.bar.setValue(this.bar.getValue() + 10);
	 		}

	 		//if(Input.isTriggered('3'))
	 		//if(Input.isTriggered('4'))
	 	}
	 }

 }
 
 
 